package cn.wisenergy.chnmuseum.party.auth.filter;

import cn.wisenergy.chnmuseum.party.auth.token.JwtToken;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authz.UnauthorizedException;
import org.apache.shiro.web.filter.authc.BasicHttpAuthenticationFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 执行流程preHandle->isAccessAllowed->isLoginAttempt->executeLogin
 */
public class JwtFilter extends BasicHttpAuthenticationFilter {

    private static final Logger LOGGER = LoggerFactory.getLogger(JwtFilter.class);

    /**
     * 判断用户是否想要登入。
     * 检测header里面是否包含Authorization字段即可
     */
    @Override
    protected boolean isLoginAttempt(ServletRequest request, ServletResponse response) {
        LOGGER.info("isLoginAttempt");
        HttpServletRequest req = (HttpServletRequest) request;
        String authorization = req.getHeader("Authorization");
        return authorization != null;
    }

    /**
     * 执行登录认证
     * 这里我们详细说明下为什么最终返回的都是true，即允许访问
     * 例如我们提供一个地址 GET /article
     * 登入用户和游客看到的内容是不同的
     * 如果在这里返回了false，请求会被直接拦截，用户看不到任何东西
     * 所以我们在这里返回true，Controller中可以通过 subject.isAuthenticated() 来判断用户是否登入
     * 如果有些资源只有登入用户才能访问，我们只需要在方法上面加上 @RequiresAuthentication 注解即可
     * 但是这样做有一个缺点，就是不能够对GET,POST等请求进行分别过滤鉴权(因为我们重写了官方的方法)，但实际上对应用影响不大
     */
    @Override
    protected boolean isAccessAllowed(ServletRequest request, ServletResponse response, Object mappedValue) {
        LOGGER.info("isAccessAllowed");
        if (isLoginAttempt(request, response)) {
        //if (!isLoginAttempt(request, response) || !executeLogin(request,response)) {
            try {
                executeLogin(request, response);
            } catch (UnauthorizedException | AuthenticationException e) {
                throw new AuthenticationException("Token失效，请重新登录", e);
            }
        }
        return true;
    }

    /**
     *
     */
    @Override
    protected boolean executeLogin(ServletRequest servletRequest, ServletResponse servletResponse) {
        LOGGER.info("executeLogin");
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        String authorization = request.getHeader("Authorization");
        if (StringUtils.isNotBlank(authorization)) {
            if (authorization.startsWith("Bearer ")) {
                authorization = authorization.substring(7);
            }
        }

        JwtToken token = new JwtToken(authorization);
        // 提交给realm进行登入，如果错误他会抛出异常并被捕获
        getSubject(servletRequest, servletResponse).login(token);
        // 如果没有抛出异常则代表登入成功，返回true
        return true;
    }

    @Override
    protected boolean onAccessDenied(ServletRequest request, ServletResponse response) {
        HttpServletResponse httpServletResponse = (HttpServletResponse) response;
        //这里是个坑，如果不设置的接受的访问源，那么前端都会报跨域错误，因为这里还没到corsConfig里面
//        httpServletResponse.setHeader("Access-Control-Allow-Origin", ((HttpServletRequest) request).getHeader("Origin"));
//        httpServletResponse.setHeader("Access-Control-Allow-Methods", "GET,POST,OPTIONS,PUT,DELETE");
//        httpServletResponse.setHeader("Access-Control-Allow-Credentials", "true");
        response401(request, response);
        return false;
    }

    /**
     * 将非法请求返回401
     */
    private void response401(ServletRequest req, ServletResponse resp) {
        try {
            HttpServletResponse response = (HttpServletResponse) resp;
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/json;charset=utf-8"); 
            response.setStatus(401);
            response.getWriter().write("{\"status\":401,\"message\":\"未登录!\"}");
        } catch (IOException e) {
            LOGGER.error(e.getMessage());
        }
    }

    /**
     * 将非法请求返回401
     */
    private void response403(ServletRequest req, ServletResponse resp) {
    	try {
    		HttpServletResponse response = (HttpServletResponse) resp;
    		response.setCharacterEncoding("UTF-8");
    		response.setContentType("application/json;charset=utf-8");
    		response.setStatus(403);
    		response.getWriter().write("{\"status\":403,\"message\":Unauthorized!}");
    	} catch (IOException e) {
    		LOGGER.error(e.getMessage());
    	}
    }

}
