package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.common.enums.AuditStatusEnum;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Add;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Update;
import cn.wisenergy.chnmuseum.party.common.vo.GenericPageParam;
import cn.wisenergy.chnmuseum.party.model.CopyrightOwner;
import cn.wisenergy.chnmuseum.party.model.ExhibitionBoard;
import cn.wisenergy.chnmuseum.party.model.ExhibitionBoardCat;
import cn.wisenergy.chnmuseum.party.service.CopyrightOwnerService;
import cn.wisenergy.chnmuseum.party.service.ExhibitionBoardCatService;
import cn.wisenergy.chnmuseum.party.service.ExhibitionBoardService;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <pre>
 * 展板分类 前端控制器
 * </pre>
 *
 * @author Danny Lee
 * @since 2021-03-19
 */
@Slf4j
@RestController
@RequestMapping("/exhibitionBoardCat")
@Api(tags = {"展板分类操作接口"})
public class ExhibitionBoardCatController extends BaseController {

    @Resource
    private ExhibitionBoardService exhibitionBoardService;
    @Resource
    private ExhibitionBoardCatService exhibitionBoardCatService;
    @Resource
    private CopyrightOwnerService copyrightOwnerService;

    @PostMapping("/save")
    @RequiresPermissions("exhibition:board:cat:save")
    @ApiOperation(value = "添加展板分类", notes = "添加展板分类")
    public Map<String, Object> saveExhibitionBoardCat(@Validated(value = {Add.class}) ExhibitionBoardCat exhibitionBoardCat) {
        // 保存业务节点信息
        boolean result = exhibitionBoardCatService.save(exhibitionBoardCat);
        // 返回操作结果
        if (result) {
            return getSuccessResult();
        } else {
            // 保存失败
            return getFailResult();
        }
    }

    @PutMapping("/update")
    @RequiresPermissions("exhibition:board:cat:update")
    @ApiOperation(value = "修改展板分类信息", notes = "修改展板分类信息")
    public Map<String, Object> updateExhibitionBoardCat(@Validated(value = {Update.class}) ExhibitionBoardCat exhibitionBoardCat) {
        boolean flag = exhibitionBoardCatService.updateById(exhibitionBoardCat);
        if (flag) {
            return getSuccessResult();
        }
        return getFailResult();
    }

    @PutMapping("/updateAuditStatus/{id}")
    @RequiresPermissions("exhibition:board:cat:update:audit:status")
    @ApiOperation(value = "更新展板分类审核状态", notes = "更新展板分类审核状态")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "id", value = "标识ID", dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "status", value = "状态", paramType = "query", dataType = "String")
    })
    public Map<String, Object> updateStatus(@NotNull(message = "展板分类ID不能为空") @PathVariable("id") String id, @RequestParam("auditStatus") AuditStatusEnum status) {
        UpdateWrapper<ExhibitionBoardCat> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", id);
        updateWrapper.eq("audit_status", status.name());
        boolean flag = exhibitionBoardCatService.update(updateWrapper);
        if (flag) {
            return getSuccessResult();
        }
        return getFailResult();
    }

    @DeleteMapping("/delete/{id}")
    @RequiresPermissions("exhibition:board:cat:delete")
    @ApiOperation(value = "根据ID删除展板分类", notes = "根据ID删除展板分类")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "id", value = "标识ID", paramType = "path", dataType = "String")
    })
    public Map<String, Object> deleteExhibitionBoardCat(@PathVariable("id") String id) {
        return getSuccessResult();
    }

    @GetMapping("/getList")
    @RequiresPermissions("exhibition:board:cat:list")
    @ApiOperation(value = "获取展板分类全部列表(无分页)", notes = "获取展板分类全部列表(无分页)")
    public Map<String, Object> getExhibitionBoardCatList(@RequestParam(value = "auditStatus", required = false) AuditStatusEnum auditStatus) {
        List<ExhibitionBoardCat> exhibitionBoardCatList = exhibitionBoardCatService.list(Wrappers.<ExhibitionBoardCat>lambdaQuery().orderByDesc(ExhibitionBoardCat::getCreateTime));
        return getResult(exhibitionBoardCatList);
    }

    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "_index", value = "分页起始偏移量", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "_size", value = "返回条数", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "nameOrCode", value = "名称或编码", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "copyrightOwner", value = "版权方", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "startDate", value = "创建时间-开始", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "endDate", value = "创建时间-结束", paramType = "query", dataType = "String")
    })
    @PostMapping("/getPageList")
    @RequiresPermissions("exhibition:board:cat:page")
    @ApiOperation(value = "获取展板分类分页列表", notes = "获取展板分类分页列表")
    public Map<String, Object> getExhibitionBoardCatPageList(GenericPageParam genericPageParam) {
        LambdaQueryWrapper<ExhibitionBoardCat> queryWrapper = new LambdaQueryWrapper<>();
        // 对名称或编码模糊查询
        if (StringUtils.isNotBlank(genericPageParam.getNameOrCode())) {
            queryWrapper.like(ExhibitionBoardCat::getName, genericPageParam.getNameOrCode());
        }
        // 根据创建时间区间检索
        if (genericPageParam.getStartDate() != null && genericPageParam.getEndDate() != null) {
            queryWrapper.ge(ExhibitionBoardCat::getCreateTime, genericPageParam.getStartDate().atTime(0, 0, 0))
                    .le(ExhibitionBoardCat::getCreateTime, genericPageParam.getEndDate().atTime(23, 59, 59));
        }
        // 设置排序规则
        queryWrapper.orderByDesc(ExhibitionBoardCat::getCreateTime);
        // 设置查询内容
        queryWrapper.select(
                ExhibitionBoardCat::getId,
                ExhibitionBoardCat::getName,
                ExhibitionBoardCat::getCreateTime,
                ExhibitionBoardCat::getUpdateTime);
        Page<ExhibitionBoardCat> page = this.exhibitionBoardCatService.page(getPage(), queryWrapper);
        for (ExhibitionBoardCat exhibitionBoardCat : page.getRecords()) {
            LambdaQueryWrapper<ExhibitionBoard> lambdaQueryWrapper = Wrappers.<ExhibitionBoard>lambdaQuery().eq(ExhibitionBoard::getExhibitionBoardCatId, exhibitionBoardCat.getId());
            List<ExhibitionBoard> exhibitionBoardList = this.exhibitionBoardService.list(lambdaQueryWrapper);
            if (!exhibitionBoardList.isEmpty()) {
                Set<String> boardCopyrightOwnerIdList = exhibitionBoardList.stream().map(ExhibitionBoard::getBoardCopyrightOwnerId).collect(Collectors.toSet());
                List<CopyrightOwner> copyrightOwnerList = this.copyrightOwnerService.listByIds(boardCopyrightOwnerIdList);
                String copyrightOwnerNames = copyrightOwnerList.stream().map(CopyrightOwner::getName).collect(Collectors.joining("、"));
                exhibitionBoardCat.setCopyrightOwnerNames(copyrightOwnerNames);
            }
        }
        return getResult(page);
    }

    @ApiOperation(value = "获取展板分类详情", notes = "获取展板分类详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "标识ID", dataType = "String", paramType = "path")
    })
    @GetMapping("/get/{id}")
    @RequiresPermissions("exhibition:board:cat:get:id")
    public Map<String, Object> getById(@PathVariable("id") String id) {
        ExhibitionBoardCat exhibitionBoardCat = exhibitionBoardCatService.getById(id);
        return getResult(exhibitionBoardCat);
    }

}

