package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.core.annotations.OperationLog;
import cn.wisenergy.chnmuseum.party.model.*;
import cn.wisenergy.chnmuseum.party.service.PermissionService;
import cn.wisenergy.chnmuseum.party.service.RolePermissionService;
import cn.wisenergy.chnmuseum.party.service.RoleService;
import cn.wisenergy.chnmuseum.party.service.impl.EmployeeRoleServiceImpl;
import cn.wisenergy.chnmuseum.party.service.impl.RoleMenuServiceImpl;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.github.stuxuhai.jpinyin.PinyinHelper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.*;

/**
 * 角色管理Controller
 */
@Api(tags = "角色管理")
@RequestMapping("/role")
@RestController
public class RoleController extends BaseController {

    private static final Logger LOGGER = LoggerFactory.getLogger(RoleController.class);

    @Autowired
    private RoleService roleService;

    @Autowired
    private PermissionService permissionService;

    @Autowired
    private RolePermissionService rolePermissionService;

    @Resource
    private EmployeeRoleServiceImpl employeeRoleService;

    @Resource
    private RoleMenuServiceImpl roleMenuService;

    /**
     * 获取单个角色
     *
     * @param Id
     * @return
     */
    @ApiOperation(value = "获取单个角色")
    @RequestMapping(value = "/getById", method = RequestMethod.GET)
    @RequiresPermissions("/role/getById")
    public ResponseEntity<Role> getById(String Id) {
        try {
            Role role = roleService.getById(Id);
            if (null == role) {
                return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
            }
            return ResponseEntity.ok(role);
        } catch (Exception e) {
            logger.error("查询错误！", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    // 获取角色分页对象
    @ApiOperation(value = "获取角色分页对象")
    @RequestMapping(value = "/getRoleList", method = RequestMethod.GET)
    @RequiresPermissions("/role/getRoleList")
    public ResponseEntity<Page<Role>> getRoleList(String name) {
        try {
            name = StringUtils.trimToNull(name);
            QueryWrapper<Role> ew = new QueryWrapper<>();
            if (StringUtils.isNotBlank(name)) {
                ew.like("name", name);
            }
            ew.eq("is_deleted", 0);
            ew.orderByAsc("sortorder");
            Page<Role> page = this.getPage();
            return ResponseEntity.ok(this.roleService.page(page, ew));
        } catch (Exception e) {
            LOGGER.error("查询角色列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 添加
     */
    @OperationLog("添加角色")
    @ApiOperation("添加角色")
    @PostMapping(value = "/add")
    @RequiresPermissions("/role/add")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "permissionIds", value = "权限id", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "menuIds", value = "菜单id", paramType = "query", dataType = "String")})
    public ResponseEntity<Map<String, Object>> add(Role role, String permissionIds) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            QueryWrapper<Role> ew = new QueryWrapper<>();
            ew.select("max(sortorder) as sort");
            Role r = this.roleService.getOne(ew);
            if (r != null) {
                role.setSortorder(r.getSortorder() + 1);
            } else {
                role.setSortorder(1);
            }
            if (StringUtils.isNoneBlank(role.getName())) {
                ew.eq("name", role.getName());
                role.setName(StringUtils.trimToNull(role.getName()));
                role.setAlias(PinyinHelper.getShortPinyin(role.getName()));
            }
            ew.or();
            if (StringUtils.isNoneBlank(role.getAlias())) {
                ew.eq("alias", role.getAlias());
                role.setAlias(StringUtils.trimToNull(role.getAlias()));
            }
            Role one = this.roleService.getOne(ew);
            if (one != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该角色已存在");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            boolean ret = false;
            role.setCreateTime(new Date(System.currentTimeMillis()));
            role.setUpdateTime(role.getCreateTime());
            ret = this.roleService.save(role);

            if (permissionIds != null) {
                String[] permissionIdsSplit = permissionIds.split(",");
                // 添加新分配的权限
                List<RolePermission> permissions = new ArrayList<RolePermission>();
                RolePermission e = null;
                for (String pid : permissionIdsSplit) {
                    e = new RolePermission();
                    e.setPid(pid);
                    e.setRid(role.getId());
                    e.setCreateTime(new Date(System.currentTimeMillis()));
                    e.setUpdateTime(e.getCreateTime());
                    permissions.add(e);
                }
                ret = rolePermissionService.saveBatch(permissions);
            }
            if (!ret) {
                resultMap.put("status", 400);
                resultMap.put("message", "添加失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "添加成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("添加角色错误!", e);
        }
        resultMap.put("status", 500);
        resultMap.put("message", "操作失败");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    @OperationLog("编辑角色")
    @ApiOperation(value = "编辑")
    @RequestMapping(value = "/edit", method = RequestMethod.PUT)
    @RequiresPermissions("/role/edit")
    public ResponseEntity<Map<String, Object>> edit(Role role) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            QueryWrapper<Role> ew = new QueryWrapper<>();
            ew.ne("id", role.getId());
            if (StringUtils.isNoneBlank(role.getName())) {
                ew.eq("name", role.getName().trim());
                role.setAlias(PinyinHelper.getShortPinyin(role.getName().trim()));
            }
            ew.or();
            if (StringUtils.isNoneBlank(role.getAlias())) {
                ew.eq("alias", role.getAlias().trim());
            }
            Role role1 = this.roleService.getOne(ew);
            if (role1 != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "角色已存在");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            role.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.roleService.updateById(role);
            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "修改失败");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            } else {
                return ResponseEntity.ok().body(null);
            }
        } catch (Exception e) {
            logger.error("修改角色错误!", e);
        }
        resultMap.put("status", 500);
        resultMap.put("message", "修改失败");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    // 刪除
    @OperationLog("删除角色")
    @ApiOperation(value = "刪除")
    @RequestMapping(value = "/delete", method = RequestMethod.DELETE)
    @RequiresPermissions("/role/delete")
    public ResponseEntity<Map<String, Object>> delete(String roleId) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            if ("1".equals(roleId) || "2".equals(roleId) || "3".equals(roleId) || "4".equals(roleId)) {
                resultMap.put("status", 400);
                resultMap.put("message", "该角色为默认角色不能被删除");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            QueryWrapper<EmployeeRole> ew = new QueryWrapper<EmployeeRole>();
            ew.eq("role_id", roleId);
            int count = this.employeeRoleService.count(ew);
            boolean ret1;
            boolean ret2;
            if (count > 0) {
                resultMap.put("status", 400);
                resultMap.put("message", "该角色下存在未删除的用户，请您删除用户后再删除角色！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            } else {
                Role role = new Role();
                role.setId(roleId);
                role.setIsDeleted(1);
                ret1 = this.roleService.removeById(role);
                RolePermission rp = new RolePermission();
                rp.setRid(roleId);
                ret2 = rolePermissionService.remove(new QueryWrapper<>(rp));
            }
            if (!ret1 && !ret2) {
                // 删除失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "删除失败！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "删除成功！");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            LOGGER.error("删除角色错误!", e);

        }
        // 500
        resultMap.put("status", 500);
        resultMap.put("message", "删除错误！");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    @RequestMapping(value = "/rights", method = RequestMethod.GET)
    @RequiresPermissions("/role/rights")
    public ResponseEntity<List<Map<String, String>>> rights(String roleId) {
        try {
            List<Permission> list = this.permissionService.list(null);
            QueryWrapper<RolePermission> wrapper = new QueryWrapper<>();
            List<Object> roleRightsList = null;
            if (roleId != null) {
                wrapper.eq("rid", roleId).select("pid");
                roleRightsList = this.rolePermissionService.listObjs(wrapper);
            }

            List<Map<String, String>> rightsList = new ArrayList<>();
            for (Permission p : list) {
                Map<String, String> map = new HashMap<>();
                map.put("id", p.getId());
                map.put("pId", p.getPid());
                map.put("name", p.getName());
                // 默认展开树
                map.put("open", "true");
                if (roleId != null && !p.getPid().equals("0")) {
                    // 如果角色已有该权限，则默认选中
                    if (roleRightsList.contains(p.getId())) {
                        map.put("checked", "true");
                    } else {
                        map.put("checked", "false");
                    }
                }
                rightsList.add(map);
            }
            return ResponseEntity.ok(rightsList);
        } catch (Exception e) {
            logger.error("获取角色权限失败", e.getMessage());
            e.printStackTrace();
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 更新角色接口权限列表
     *
     * @return
     * @Params roleId
     */
    @OperationLog("更新角色操作权限")
    @ApiOperation(value = "更新权限列表")
    @RequestMapping(value = "/updateRoleRights", method = RequestMethod.POST)
    @RequiresPermissions("/role/updateRoleRights")
    public ResponseEntity<Map<String, Object>> updateRoleRights(String roleId, String permissionIds) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            if (roleId != null) {
                // 查询出本角色已经分配了的权限
                RolePermission rolePermission = new RolePermission();
                rolePermission.setRid(roleId);
                QueryWrapper<RolePermission> ew = new QueryWrapper<>();
                ew.setEntity(rolePermission);
                List<RolePermission> roleRightList = this.rolePermissionService.list(ew);
                boolean ret = false;
                // 如果存在权限，先进行删除
                if (roleRightList.size() > 0) {
                    this.rolePermissionService.remove(ew);
                }

                String[] rightIds = StringUtils.split(permissionIds, ",");
                if (StringUtils.isNotBlank(permissionIds) && rightIds != null) {
                    // 添加新分配的权限
                    List<RolePermission> permissions = new ArrayList<RolePermission>();
                    RolePermission e;
                    for (String pid : rightIds) {
                        e = new RolePermission();
                        e.setPid(pid);
                        e.setRid(roleId);
                        e.setCreateTime(new Date(System.currentTimeMillis()));
                        e.setUpdateTime(e.getCreateTime());
                        permissions.add(e);
                    }
                    ret = rolePermissionService.saveBatch(permissions);
                } else {
                    resultMap.put("status", 400);
                    resultMap.put("message", "错误请求！");
                    return ResponseEntity.status(HttpStatus.OK).body(resultMap);
                }
                if (!ret) {
                    // 更新失败, 500
                    resultMap.put("status", 500);
                    resultMap.put("message", "更新失败！");
                    return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
                }
                // 204
                resultMap.put("status", 200);
                resultMap.put("message", "更新成功！");
                return ResponseEntity.status(HttpStatus.OK).body(resultMap);
            } else {
                resultMap.put("status", 400);
                resultMap.put("message", "请选择角色！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
        } catch (Exception e) {
            logger.error("更新角色权限错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    /**
     * 更新角色菜单列表
     *
     * @return
     */
    @OperationLog("更新角色菜单权限")
    @ApiOperation(value = "更新菜单列表")
    @RequestMapping(value = "/updateRoleMenu", method = RequestMethod.POST)
    @RequiresPermissions("/role/updateRoleMenu")
    public ResponseEntity<Map<String, Object>> updateRoleMenu(String roleId, String menuIds) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            if (roleId != null) {
                // 查询出本角色已经分配了的菜单
                RoleMenu roleMenu = new RoleMenu();
                roleMenu.setRoleId(roleId);
                QueryWrapper<RoleMenu> ew = new QueryWrapper<>();
                ew.setEntity(roleMenu);
                List<RoleMenu> roleMenuList = this.roleMenuService.list(ew);
                boolean ret = false;
                // 如果存在菜单，先进行删除
                if (roleMenuList.size() > 0) {
                    this.roleMenuService.remove(ew);
                }

                String[] mIds = StringUtils.split(menuIds, ",");
                if (StringUtils.isNotBlank(menuIds) && mIds != null) {
                    // 添加新分配的菜单
                    List<RoleMenu> menus = new ArrayList<RoleMenu>();
                    RoleMenu e = null;
                    for (String mid : mIds) {
                        e = new RoleMenu();
                        e.setMenuId(mid);
                        e.setRoleId(roleId);
                        e.setCreateTime(new Date(System.currentTimeMillis()));
                        e.setUpdateTime(e.getCreateTime());
                        menus.add(e);
                    }
                    ret = roleMenuService.saveBatch(menus);
                    if (!ret) {
                        // 更新失败, 500
                        resultMap.put("status", 500);
                        resultMap.put("message", "更新失败！");
                        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
                    }
                }
                // 204
                resultMap.put("status", 200);
                resultMap.put("message", "更新成功！");
                return ResponseEntity.status(HttpStatus.OK).body(resultMap);
            } else {
                resultMap.put("status", 400);
                resultMap.put("message", "请选择角色！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
        } catch (Exception e) {
            logger.error("更新角色菜单错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    @OperationLog("启用角色")
    @ApiOperation(value = "启动")
    @RequestMapping(value = "/enableRole", method = RequestMethod.PUT)
    @RequiresPermissions("/role/enableRole")
    public ResponseEntity<Map<String, Object>> enableRole(String roleId) {
        try {
            Map<String, Object> map = new HashMap<>();
            Role entity = new Role();
            entity.setId(roleId);
            entity.setStatus(true);
            entity.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.roleService.updateById(entity);
            if (!ret) {
                map.put("status", "500");
                map.put("message", "服务器错误");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(map);
            }
            map.put("status", "201");
            map.put("message", "启用成功");
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("用户启动出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(
                null);
    }

    // 批量启动
    @OperationLog("批量启用角色")
    @ApiOperation(value = "批量启动")
    @RequestMapping(value = "/batchEnable", method = RequestMethod.PUT)
    @RequiresPermissions("/role/batchEnable")
    public ResponseEntity<Map<String, Object>> batchEnable(String roleIds) {
        try {
            Map<String, Object> map = new HashMap<>();
            boolean flag = false;
            if (StringUtils.isNotBlank(roleIds)) {
                String[] roleIdArr = roleIds.split(",");
                Role entity = new Role();
                entity.setStatus(true);
                entity.setUpdateTime(new Date(System.currentTimeMillis()));
                for (String roleId : roleIdArr) {
                    entity.setId(roleId);
                    flag = this.roleService.updateById(entity);
                    if (!flag) {
                        break;
                    }
                }
            } else {
                map.put("status", "400");
                map.put("message", "请选择要启动的用户");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!flag) {
                map.put("status", "400");
                map.put("message", "批量启动失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            } else {
                map.put("status", "201");
                map.put("message", "批量启动成功");
                return ResponseEntity.status(HttpStatus.OK).body(map);
            }
        } catch (Exception e) {
            logger.error("批量启动失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    /**
     * 禁用
     *
     * @param roleId
     * @return
     */
    @OperationLog("禁用角色")
    @ApiOperation(value = "禁用")
    @RequestMapping(value = "/disable", method = RequestMethod.PUT)
    @RequiresPermissions("/role/disable")
    public ResponseEntity<Map<String, Object>> disableRole(String roleId) {
        try {
            Map<String, Object> map = new HashMap<>();
            Role entity = new Role();
            entity.setId(roleId);
            entity.setStatus(false);
            entity.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.roleService.updateById(entity);

            if (!ret) {
                map.put("status", "400");
                map.put("message", "禁用错误");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            map.put("status", "201");
            map.put("message", "禁用成功");
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("禁用用户出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(
                null);
    }

    // 批量禁用
    @OperationLog("批量禁用角色")
    @ApiOperation(value = "批量禁用")
    @RequestMapping(value = "/batchDis", method = RequestMethod.PUT)
    @RequiresPermissions("/role/batchDis")
    public ResponseEntity<Map<String, Object>> batchDis(String roleIds) {
        try {
            Map<String, Object> map = new HashMap<>();
            boolean flag = false;
            if (StringUtils.isNotBlank(roleIds)) {
                String[] roleIdArr = roleIds.split(",");
                Role entity = new Role();
                entity.setStatus(false);
                entity.setUpdateTime(new Date(System.currentTimeMillis()));
                for (String roleId : roleIdArr) {
                    entity.setId(roleId);
                    flag = this.roleService.updateById(entity);
                    if (!flag) {
                        break;
                    }
                }
            } else {
                map.put("status", "400");
                map.put("message", "请选择要禁用的用户");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!flag) {
                map.put("status", "400");
                map.put("message", "禁用失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            } else {
                map.put("status", "201");
                map.put("message", "禁用成功");
                return ResponseEntity.status(HttpStatus.OK).body(map);
            }
        } catch (Exception e) {
            logger.error("用户禁用失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 批量删除
    @OperationLog("批量删除角色")
    @ApiOperation(value = "批量删除")
    @RequestMapping(value = "/batchDel", method = RequestMethod.DELETE)
    @RequiresPermissions("/role/batchDel")
    public ResponseEntity<Map<String, Object>> batchDel(String roleIds) {
        try {
            Map<String, Object> map = new HashMap<>();
            boolean flag = false;
            boolean batchDelflag = false;
            String selectRoleIds = "";
            if (StringUtils.isNotBlank(roleIds)) {
                String[] roleIdArr = roleIds.split(",");
                Role entity = new Role();
                entity.setIsDeleted(1);
                for (String roleId : roleIdArr) {
                    if ("1".equals(roleId) || "2".equals(roleId) || "3".equals(roleId) || "4".equals(roleId)) {
                        if ("".equals(selectRoleIds)) {
                            selectRoleIds = roleId;
                        } else {
                            selectRoleIds += "," + roleId;
                        }
                        continue;
                    }
                    entity.setId(roleId);
                    this.roleService.updateById(entity);
                    //同时删除角色的权限
                    RolePermission rp = new RolePermission();
                    rp.setRid(roleId);
                    flag = rolePermissionService.remove(new QueryWrapper<>(rp));

                    if (!flag) {
                        batchDelflag = true;
                        break;
                    }
                }
            } else {
                map.put("status", "400");
                map.put("message", "请选择要删除的角色");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }

            if (batchDelflag) {
                map.put("status", 400);
                map.put("message", "删除失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!"".equals(selectRoleIds) && !flag) {
                map.put("status", 400);
                map.put("message", "系统默认角色不能被删除");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }

            map.put("status", 200);
            String message = "删除成功";
            if (!"".equals(selectRoleIds) && flag) {
                message = "系统默认角色不能被删除，其他选中角色删除成功";
            }
            map.put("message", message);
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("角色删除失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 获取角色下拉框数据
    @ApiOperation(value = "获取角色下拉框数据")
    @RequestMapping(value = "/getRoleBox", method = RequestMethod.GET)
    @RequiresPermissions("/role/getRoleBox")
    public Map<String,Object> getRoleBox() {
        try {
            QueryWrapper<Role> ew = new QueryWrapper<>();
            ew.select("id", "name");
            ew.eq("is_deleted", 0);
            ew.eq("status", 1);
            ew.orderByAsc("sortorder");
            ew.orderByDesc("create_time");
            List<Role> list = roleService.list(ew);
            return getResult(list);
        } catch (Exception e) {
            LOGGER.error("查询角色列表出错!", e);
        }
        return getFailResult();
    }
}
