package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.common.dfs.FastDFSUtils;
import cn.wisenergy.chnmuseum.party.common.enums.*;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Add;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Update;
import cn.wisenergy.chnmuseum.party.common.vo.GenericPageParam;
import cn.wisenergy.chnmuseum.party.model.*;
import cn.wisenergy.chnmuseum.party.service.*;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <pre>
 * 视频内容 前端控制器
 * </pre>
 *
 * @author Danny Lee
 * @since 2021-03-16
 */
@Slf4j
@RestController
@RequestMapping("/videoContent")
@Api(tags = {"视频内容内容接口"})
public class VideoContentController extends BaseController {

    @Resource
    private VideoContentService videoContentService;
    @Resource
    private CopyrightOwnerService copyrightOwnerService;
    @Resource
    private VideoContentCatService videoContentCatService;
    @Resource
    private AssetService assetService;
    @Resource
    private AuditService auditService;

    @PostMapping(value = "/save")
    @RequiresPermissions("video:content:save")
    @ApiOperation(value = "添加视频内容", notes = "添加视频内容")
    public Map<String, Object> saveAsset(@Validated(value = {Add.class}) VideoContent videoContent) {
        final List<String> videoFileIdList = videoContent.getVideoFileIdList();
        if (videoFileIdList == null || videoFileIdList.isEmpty()) {
            return getFailResult("400", "视频文件必须上传");
        }
        videoContent.setAuditStatus(AuditStatusEnum.TBC.name());
        videoContent.setPublished(false);
        videoContent.setDeleted(false);
        // 保存业务节点信息
        boolean result = videoContentService.save(videoContent);
        // 返回操作结果
        if (result) {
            for (String videoFileId : videoFileIdList) {
                final Asset asset = this.assetService.getById(videoFileId);
                asset.setThumbnail(videoContent.getThumbnail());
                asset.setFileType(FileTypeEnum.VIDEO.name());
                asset.setFileCat(FileCatEnum.VIDEO_CONTENT.name());
                asset.setRefItemId(videoContent.getId());
                this.assetService.updateById(asset);
            }

            final Audit audit = Audit.builder()
                    .content(videoContent.getName())
                    .refItemId(videoContent.getId())
                    .type(AuditTypeEnum.VIDEO_CONTENT.name())
                    .operation(AuditOperationEnum.ADD.name())
                    .status(AuditStatusEnum.TBC.name())
                    .deleted(false)
                    .level(AuditStatusEnum.TBC.name())
                    .build();
            this.auditService.save(audit);
            return getSuccessResult();
        }
        return getFailResult();
    }

    @PutMapping("/update")
    @RequiresPermissions("video:content:update")
    @ApiOperation(value = "修改视频内容信息", notes = "修改视频内容信息")
    public Map<String, Object> updateAsset(@Validated(value = {Update.class}) VideoContent videoContent) {
        videoContent.setAuditStatus(AuditStatusEnum.TBC.name());
        videoContent.setPublished(false);
        videoContent.setDeleted(false);
        boolean flag = videoContentService.updateById(videoContent);
        if (flag) {
            final List<String> videoFileIdList = videoContent.getVideoFileIdList();
            if (videoFileIdList != null && !videoFileIdList.isEmpty()) {
                final LambdaQueryWrapper<Asset> assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, videoContent.getId());
                final List<Asset> assetList = this.assetService.list(assetQueryWrapper);
                final Map<String, String> collect = assetList.stream().collect(Collectors.toMap(Asset::getId, Asset::getFileUrl));
                for (String videoFileId : videoFileIdList) {
                    final Asset asset = this.assetService.getById(videoFileId);
                    asset.setThumbnail(videoContent.getThumbnail());
                    asset.setFileType(FileTypeEnum.VIDEO.name());
                    asset.setFileCat(FileCatEnum.VIDEO_CONTENT.name());
                    asset.setRefItemId(videoContent.getId());
                    this.assetService.updateById(asset);
                    collect.remove(videoFileId);
                }
                collect.forEach((k, v) -> {
                    boolean deleted = this.assetService.removeById(k);
                    if (deleted) {
                        FastDFSUtils.deleteFile(v);
                    }
                });
            }

            final Audit audit = Audit.builder()
                    .content(videoContent.getName())
                    .refItemId(videoContent.getId())
                    .type(AuditTypeEnum.VIDEO_CONTENT.name())
                    .operation(AuditOperationEnum.EDIT.name())
                    .status(AuditStatusEnum.TBC.name())
                    .deleted(false)
                    .level(AuditStatusEnum.TBC.name())
                    .build();
            this.auditService.save(audit);

            return getSuccessResult();
        }
        return getFailResult();
    }

    @DeleteMapping("/delete/{id}")
    @RequiresPermissions("video:content:delete")
    @ApiOperation(value = "根据ID删除视频内容", notes = "根据ID删除视频内容")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "id", value = "标识ID", paramType = "path", dataType = "String")
    })
    public Map<String, Object> deleteAsset(@PathVariable("id") String id) {
        final Audit audit = Audit.builder()
                .content(this.videoContentService.getById(id).getName())
                .refItemId(id)
                .type(AuditTypeEnum.VIDEO_CONTENT.name())
                .operation(AuditOperationEnum.REMOVE.name())
                .status(AuditStatusEnum.TBC.name())
                .deleted(false)
                .level(AuditStatusEnum.TBC.name())
                .build();
        final boolean result = this.auditService.save(audit);
        if (result) {
            return getSuccessResult();
        }
        return getFailResult();
    }

    @GetMapping("/getList")
    @RequiresPermissions("video:content:list")
    @ApiOperation(value = "获取视频内容全部列表(无分页)", notes = "获取视频内容全部列表(无分页)")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "auditStatus", value = "审核状态", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "videoContentCatId", value = "视频内容分类ID", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "videoContentCopyrightOwnerId", value = "视频内容版权方ID", paramType = "query", dataType = "String")
    })
    public Map<String, Object> getVideoContentList(@RequestParam(value = "auditStatus", required = false) AuditStatusEnum auditStatus,
                                                   @RequestParam(value = "videoContentCatId", required = false) String videoContentCatId,
                                                   @RequestParam(value = "videoContentCopyrightOwnerId", required = false) String videoContentCopyrightOwnerId) {
        final LambdaQueryWrapper<VideoContent> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        //lambdaQueryWrapper.eq(Asset::getPublished, true);
        if (auditStatus != null) {
            lambdaQueryWrapper.eq(VideoContent::getAuditStatus, auditStatus.name());
        }
        if (StringUtils.isNotBlank(videoContentCatId)) {
            lambdaQueryWrapper.eq(VideoContent::getVideoContentCatId, videoContentCatId);
        }
        if (StringUtils.isNotBlank(videoContentCopyrightOwnerId)) {
            lambdaQueryWrapper.eq(VideoContent::getVideoContentCopyrightOwnerId, videoContentCopyrightOwnerId);
        }
        lambdaQueryWrapper.orderByDesc(VideoContent::getCreateTime);
        List<VideoContent> videoContentList = videoContentService.list(lambdaQueryWrapper);
        return getResult(videoContentList);
    }

    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "_index", value = "分页起始偏移量", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "_size", value = "返回条数", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "nameOrCode", value = "名称或编码", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "videoContentCatId", value = "视频内容分类ID", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "videoContentCopyrightOwnerId", value = "视频内容版权方ID", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "startDate", value = "创建时间-开始", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "endDate", value = "创建时间-结束", paramType = "query", dataType = "String")
    })
    @PostMapping("/getPageList")
    @RequiresPermissions("video:content:page")
    @ApiOperation(value = "获取视频内容分页列表", notes = "获取视频内容分页列表")
    public Map<String, Object> getAssetPageList(GenericPageParam genericPageParam) {
        LambdaQueryWrapper<VideoContent> queryWrapper = new LambdaQueryWrapper<>();
        // 对名称或编码模糊查询
        if (StringUtils.isNotBlank(genericPageParam.getNameOrCode())) {
            queryWrapper.like(VideoContent::getName, genericPageParam.getNameOrCode());
        }
        if (StringUtils.isNotBlank(genericPageParam.getVideoContentCatId())) {
            queryWrapper.eq(VideoContent::getVideoContentCatId, genericPageParam.getVideoContentCatId());
        }
        if (StringUtils.isNotBlank(genericPageParam.getVideoContentCopyrightOwnerId())) {
            queryWrapper.eq(VideoContent::getVideoContentCopyrightOwnerId, genericPageParam.getVideoContentCopyrightOwnerId());
        }
        // 根据创建时间区间检索
        if (genericPageParam.getStartDate() != null && genericPageParam.getEndDate() != null) {
            queryWrapper.ge(VideoContent::getCreateTime, genericPageParam.getStartDate().atTime(0, 0, 0))
                    .le(VideoContent::getCreateTime, genericPageParam.getEndDate().atTime(23, 59, 59));
        }
        // 设置排序规则
        queryWrapper.orderByDesc(VideoContent::getCreateTime);
        // 设置查询内容
        queryWrapper.select(
                VideoContent::getId,
                VideoContent::getName,
                VideoContent::getAuditStatus,
                VideoContent::getPublished,
                VideoContent::getDeleted,
                VideoContent::getVideoContentCatId,
                VideoContent::getVideoContentCopyrightOwnerId,
                VideoContent::getCreateTime,
                VideoContent::getUpdateTime);
        Page<VideoContent> page = this.videoContentService.page(getPage(), queryWrapper);
        for (VideoContent videoContent : page.getRecords()) {
            if (videoContent.getVideoContentCatId() != null) {
                VideoContentCat videoContentCat = this.videoContentCatService.getById(videoContent.getVideoContentCatId());
                videoContent.setVideoContentCatName(videoContentCat.getName());
            }
            if (videoContent.getVideoContentCopyrightOwnerId() != null) {
                CopyrightOwner copyrightOwner = this.copyrightOwnerService.getById(videoContent.getVideoContentCopyrightOwnerId());
                videoContent.setVideoContentCopyrightOwnerName(copyrightOwner.getName());
            }
        }
        return getResult(page);
    }

    @ApiOperation(value = "获取视频内容详情", notes = "获取视频内容详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "标识ID", dataType = "String", paramType = "path", required = true)
    })
    @GetMapping("/get/{id}")
    @RequiresPermissions("video:content:get:id")
    public Map<String, Object> getById(@PathVariable("id") String id) {
        VideoContent videoContent = videoContentService.getById(id);
        if (videoContent.getVideoContentCatId() != null) {
            VideoContentCat videoContentCat = this.videoContentCatService.getById(videoContent.getVideoContentCatId());
            if (videoContentCat != null) {
                videoContent.setVideoContentCatName(videoContentCat.getName());
            }
        }
        if (videoContent.getVideoContentCopyrightOwnerId() != null) {
            CopyrightOwner copyrightOwner = this.copyrightOwnerService.getById(videoContent.getVideoContentCopyrightOwnerId());
            if (copyrightOwner != null) {
                videoContent.setVideoContentCopyrightOwnerName(copyrightOwner.getName());
            }
        }
        final LambdaQueryWrapper<Asset> assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, id);
        assetQueryWrapper.eq(Asset::getFileCat, FileCatEnum.VIDEO_CONTENT.name());
        final List<Asset> videoFileList = this.assetService.list(assetQueryWrapper);
        videoContent.setVideoFileList(videoFileList);

        final LambdaQueryWrapper<Audit> auditQueryWrapper = Wrappers.<Audit>lambdaQuery().eq(Audit::getRefItemId, id);
        auditQueryWrapper.select(Audit::getContent);
        auditQueryWrapper.select(Audit::getType);
        auditQueryWrapper.select(Audit::getOperation);
        auditQueryWrapper.select(Audit::getStatus);
        auditQueryWrapper.select(Audit::getFirstTime);
        auditQueryWrapper.select(Audit::getFirstRemarks);
        auditQueryWrapper.select(Audit::getSecondTime);
        auditQueryWrapper.select(Audit::getSecondTime);
        auditQueryWrapper.select(Audit::getLevel);
        final List<Audit> auditList = this.auditService.list(auditQueryWrapper);
        videoContent.setAuditHistoryList(auditList);

        return getResult(videoContent);
    }

}

