package cn.wisenergy.chnmuseum.party.common.dfs;

import cn.wisenergy.chnmuseum.party.common.mvc.InterfaceException;
import cn.wisenergy.chnmuseum.party.common.util.CopyStreamUtils;
import cn.wisenergy.chnmuseum.party.common.util.FileTypeUtil;
import cn.wisenergy.chnmuseum.party.common.util.FileUtil;
import com.github.tobato.fastdfs.domain.fdfs.FileInfo;
import com.github.tobato.fastdfs.domain.fdfs.MetaData;
import com.github.tobato.fastdfs.domain.fdfs.StorePath;
import com.github.tobato.fastdfs.domain.fdfs.ThumbImageConfig;
import com.github.tobato.fastdfs.domain.proto.storage.DownloadByteArray;
import com.github.tobato.fastdfs.exception.FdfsServerException;
import com.github.tobato.fastdfs.service.FastFileStorageClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

@Slf4j
@Component
public class FastDFSUtils {

    private static String dfsFileAccessBasePath;
    private static ThumbImageConfig imageConfig;
    private static FastFileStorageClient storageClient;

    @Value("${dfsFileAccessBasePath:#{null}}")
    public void setDfsFileAccessBasePath(String dfsFileAccessBasePath) {
        FastDFSUtils.dfsFileAccessBasePath = dfsFileAccessBasePath;
    }

    @Resource
    private ThumbImageConfig thumbImageConfig;
    @Resource
    private FastFileStorageClient fastFileStorageClient;

    @PostConstruct
    public void init() {
        FastDFSUtils.storageClient = fastFileStorageClient;
        FastDFSUtils.imageConfig = thumbImageConfig;
    }

    public static Map<String, Object> uploadUeImage(InputStream inputStream, long fileSize, String fileName) {
        boolean isImage = FileTypeUtil.isImageByExtension(fileName);
        if (isImage) {
            String mimeType = FileUtil.getMimeType(fileName);
            final String storePath = FastDFSUtils.uploadFile(inputStream, fileSize, fileName, null);
            Map<String, Object> uploadResult = new HashMap<>();
            uploadResult.put("url", storePath);
            uploadResult.put("status", true);
            uploadResult.put("message", "文件上传成功！");
            uploadResult.put("title", fileName);
            uploadResult.put("mimeType", mimeType);
            uploadResult.put("size", fileSize);
            return uploadResult;
        }
        throw new InterfaceException("400", "文件不是图片类型");
    }

    public static String uploadFile(InputStream inputStream, long size, String fileName, Set<MetaData> metaDataSet) {
        final StorePath storePath = storageClient.uploadFile(inputStream, size, FilenameUtils.getExtension(fileName), metaDataSet);
        return dfsFileAccessBasePath + "/" + storePath.getFullPath();
    }

    public static String uploadVideo(InputStream inputStream, long size, String fileName, Set<MetaData> metaDataSet) {
        Map<String, Object> map = CopyStreamUtils.copyInputStream(inputStream);
        String md5 = (String) map.get("md5");
        metaDataSet.add(new MetaData("MD5", md5));
        InputStream is = (InputStream) map.get("inputStream");
        final StorePath storePath = storageClient.uploadFile(is, size, FilenameUtils.getExtension(fileName), metaDataSet);
        return dfsFileAccessBasePath + "/" + storePath.getFullPath();
    }

    /**
     * 使用 FastDFS 提供的客户端 storageClient 来进行文件上传，最后将上传结果返回。
     * 根据 groupName 和文件名获取文件信息。
     */
    public static FileInfo getFileInfo(String fileUrl) {
        fileUrl = fileUrl.replace(dfsFileAccessBasePath + "/", "");
        String groupName = fileUrl.substring(0, fileUrl.indexOf("/"));
        String path = fileUrl.substring(fileUrl.indexOf("/") + 1);
        return storageClient.queryFileInfo(groupName, path);
    }

    /**
     * 根据 groupName 和文件名获取文件信息。
     */
    public static Set<MetaData> getFileMetaData(String fileUrl) {
        fileUrl = fileUrl.replace(dfsFileAccessBasePath + "/", "");
        String groupName = fileUrl.substring(0, fileUrl.indexOf("/"));
        String path = fileUrl.substring(fileUrl.indexOf("/") + 1);
        return storageClient.getMetadata(groupName, path);
    }

    /**
     * 下载文件(字节数组)
     */
    public static byte[] downloadFile(String fileUrl) {
        fileUrl = fileUrl.replace(dfsFileAccessBasePath + "/", "");
        String groupName = fileUrl.substring(0, fileUrl.indexOf("/"));
        String path = fileUrl.substring(fileUrl.indexOf("/") + 1);
        return storageClient.downloadFile(groupName, path, new DownloadByteArray());
    }

    /**
     * 下载文件
     *
     * @param groupName
     * @param path
     * @return
     */
    public static InputStream deleteFile(String groupName, String path) {
        try {
            return storageClient.downloadFile(groupName, path, inputStream -> inputStream);
        } catch (FdfsServerException e) {
            //不起作用
            log.error("文件不存在，下载失败：" + e.getErrorCode());
            throw new InterfaceException("文件不存在，下载失败：" + e.getErrorCode());
        }
    }

    /**
     * 删除文件
     */
    public static byte[] deleteFile(String fileUrl) {
        fileUrl = fileUrl.replace(dfsFileAccessBasePath + "/", "");
        String groupName = fileUrl.substring(0, fileUrl.indexOf("/"));
        String path = fileUrl.substring(fileUrl.indexOf("/") + 1);
        return storageClient.downloadFile(groupName, path, new DownloadByteArray());
    }

    private static int[] getImageInfo(byte[] bytes) {
        try {
            ByteArrayInputStream baos = new ByteArrayInputStream(bytes);
            BufferedImage image = ImageIO.read(baos);
            int width = image.getWidth();
            int height = image.getHeight();
            return new int[]{width, height};
        } catch (Exception e) {
            log.error("FastDFSUtils.getImageInfo() error:", e);
        }
        return new int[]{0, 0};
    }

    private static ByteArrayOutputStream translateToByteArray(InputStream inputStream) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        IOUtils.copy(inputStream, out);
        return out;
    }

//    private static class UploadFileSender implements UploadCallback {
//        private final InputStream is;
//
//        public UploadFileSender(InputStream is) {
//            this.is = is;
//        }
//
//        @Override
//        public int send(OutputStream out) throws IOException {
//            byte[] b = new byte[1024];
//            int readBytes;
//            while ((readBytes = is.read(b)) != -1) {
//                out.write(b, 0, readBytes);
//            }
//            return 0;
//        }
//    }

}
