package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.common.validator.groups.Add;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Update;
import cn.wisenergy.chnmuseum.party.common.vo.GenericPageParam;
import cn.wisenergy.chnmuseum.party.model.LearningContent;
import cn.wisenergy.chnmuseum.party.model.LearningProject;
import cn.wisenergy.chnmuseum.party.service.LearningContentService;
import cn.wisenergy.chnmuseum.party.service.LearningProjectService;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <pre>
 *  前端控制器
 * </pre>
 *
 * @author Danny Lee
 * @since 2021-03-26
 */
@Slf4j
@RestController
@RequestMapping("/learningProject")
@Api(tags = {"学习项目操作接口"})
public class LearningProjectController extends BaseController {

    @Resource
    private LearningProjectService learningProjectService;
    @Resource
    private LearningContentService learningContentService;

    @PostMapping("/save")
    @RequiresPermissions("learning:project:save")
    @ApiOperation(value = "添加", notes = "添加")
    public Map<String, Object> saveLearningProject(@Validated(value = {Add.class}) LearningProject learningProject) {
        // 保存业务节点信息
        boolean result = learningProjectService.save(learningProject);
        // 返回操作结果
        if (result) {
            return getSuccessResult();
        } else {
            // 保存失败
            return getFailResult();
        }
    }

    @PutMapping("/update")
    @RequiresPermissions("learning:project:update")
    @ApiOperation(value = "修改信息", notes = "修改信息")
    public Map<String, Object> updateLearningProject(@Validated(value = {Update.class}) LearningProject learningProject) {
        boolean flag = learningProjectService.updateById(learningProject);
        if (flag) {
            return getSuccessResult();
        }
        return getFailResult();
    }

    @DeleteMapping("/delete/{id}")
    @RequiresPermissions("learning:project:delete")
    @ApiOperation(value = "根据ID下架学习项目", notes = "根据ID下架学习项目")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "id", value = "标识ID", paramType = "path", dataType = "String")
    })
    public Map<String, Object> deleteLearningProject(@PathVariable("id") String id) {
        return getSuccessResult();
    }

    @GetMapping("/getList")
    @RequiresPermissions("learning:project:list")
    @ApiOperation(value = "获取全部列表(无分页)", notes = "获取全部列表(无分页)")
    public Map<String, Object> getLearningProjectList() {
        List<LearningProject> learningProjectList = learningProjectService.list(Wrappers.<LearningProject>lambdaQuery().orderByDesc(LearningProject::getCreateTime));
        return getResult(learningProjectList);
    }

    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "_index", value = "分页起始偏移量", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "_size", value = "返回条数", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "nameOrCode", value = "名称或编码", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "startDate", value = "创建时间-开始", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "endDate", value = "创建时间-结束", paramType = "query", dataType = "String")
    })
    @PostMapping("/getPageList")
    @RequiresPermissions("learning:project:page")
    @ApiOperation(value = "获取分页列表", notes = "获取分页列表")
    public Map<String, Object> getLearningProjectPageList(GenericPageParam genericPageParam) {
        LambdaQueryWrapper<LearningProject> queryWrapper = new LambdaQueryWrapper<>();
        // 对名称或编码模糊查询
        if (StringUtils.isNotBlank(genericPageParam.getNameOrCode())) {
            queryWrapper.like(LearningProject::getName, genericPageParam.getNameOrCode());
        }
        // 根据创建时间区间检索
        if (genericPageParam.getStartDate() != null && genericPageParam.getEndDate() != null) {
            queryWrapper.ge(LearningProject::getCreateTime, genericPageParam.getStartDate().atTime(0, 0, 0))
                    .le(LearningProject::getCreateTime, genericPageParam.getEndDate().atTime(23, 59, 59));
        }
        // 设置排序规则
        queryWrapper.orderByDesc(LearningProject::getCreateTime);
        // 设置查询内容
        queryWrapper.select(
                LearningProject::getId,
                LearningProject::getName,
                LearningProject::getRemarks,
                LearningProject::getCreateTime,
                LearningProject::getUpdateTime);
        Page<LearningProject> page = this.learningProjectService.page(getPage(), queryWrapper);
        for (LearningProject learningProject : page.getRecords()) {
            LambdaQueryWrapper<LearningContent> lambdaQueryWrapper = Wrappers.<LearningContent>lambdaQuery()
                    .eq(LearningContent::getLearningProjectId, learningProject.getId())
                    .eq(LearningContent::getPublished, true);
            lambdaQueryWrapper.select(LearningContent::getName);
            List<LearningContent> learningContentList = this.learningContentService.list(lambdaQueryWrapper);
            String learningContentNames = learningContentList.stream().map(LearningContent::getName).collect(Collectors.joining("、"));
            learningProject.setLearningContentNames(learningContentNames);
        }
        return getResult(page);
    }

    @ApiOperation(value = "获取详情", notes = "获取详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "标识ID", dataType = "String", paramType = "path")
    })
    @GetMapping("/get/{id}")
    @RequiresPermissions("learning:project:get:id")
    public Map<String, Object> getById(@PathVariable("id") String id) {
        LearningProject learningProject = learningProjectService.getById(id);
        return getResult(learningProject);
    }

}

