package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.core.annotations.OperationLog;
import cn.wisenergy.chnmuseum.party.model.*;
import cn.wisenergy.chnmuseum.party.service.PublicService;
import cn.wisenergy.chnmuseum.party.service.impl.*;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.*;

/**
 * <p>
 * 网点信息 前端控制器
 * </p>
 *
 * @author 杨智平
 * @since 2018-08-29
 */
@Api(tags = "网点信息")
@RestController
@RequestMapping("/bankBranchInfo")
public class BankBranchInfoController extends BaseController {

    private static final Logger logger = LoggerFactory.getLogger(BankBranchInfoController.class);

    @Resource
    private BankBranchInfoServiceImpl bankBranchInfoService;

    @Resource
    private EmployeeServiceImpl employeeService;

    @Value("${IMAGE_BASE_URL}")
    private String IMAGE_BASE_URL;

    @Resource
    private HotSpotServiceImpl hotSpotService;

    @Resource
    private DemandInfoServiceImpl demandInfoService;

    @Resource
    private BusinessInfoServiceImpl businessInfoService;

    @Resource
    private HotProductActivityServiceImpl hotProductActivityService;

    @Resource
    private ProductAppointmentRecordServiceImpl productAppointmentRecordService;

    @Resource
    private PublicService publicService;

    /**
     * 获取单个网点信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取单个网点信息")
    @GetMapping(value = "/getById")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/getById")
    public ResponseEntity<BankBranchInfo> getById(String id) {
        try {
            BankBranchInfo one = this.bankBranchInfoService.getById(id);
            if (one.getCreator() != null && !"".equals(one.getCreator())) {
                Employee creatorName = this.employeeService.getById(one.getCreator());
                if (creatorName != null) {
                    one.setCreatorName(creatorName.getName());
                }
            }
            if (one.getModifier() != null && !"".equals(one.getModifier())) {
                Employee modifierName = this.employeeService.getById(one.getModifier());
                if (modifierName != null) {
                    one.setModifierName(modifierName.getName());
                }
            }
            if (null != one) {
                return ResponseEntity.ok(one);
            }
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
        } catch (Exception e) {
            logger.error("获取单个网点信息出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * App获取单个网点信息
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "App获取单个网点信息")
    @GetMapping(value = "/getByIdOnApp")
    public ResponseEntity<BankBranchInfo> getByIdOnApp(String id, String userId, String bankId) {
        try {
            BankBranchInfo one = this.bankBranchInfoService.getById(id);
            if (null != one) {
                //将数据放入缓存
                this.publicService.cacheData(id, userId, bankId, 2);
                return ResponseEntity.ok(one);
            }
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
        } catch (Exception e) {
            logger.error("获取单个网点信息出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 获取网点列表
     *
     * @param name
     * @return
     */
    @ApiOperation(value = "获取网点列表")
    @GetMapping(value = "/getList")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/getList")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "name", value = "网点名称", dataType = "String")})
    public ResponseEntity<Page<BankBranchInfo>> getList(String name, String roleId, String currentBankId, String query) {
        try {
            Page<BankBranchInfo> page = this.getPage();
            if (!"3".equals(roleId)) {
                roleId = "0";
            }
            Page<BankBranchInfo> bankBranchPage = this.bankBranchInfoService.selectListByName(page, name, roleId, currentBankId, query);
            return ResponseEntity.ok(bankBranchPage);
        } catch (Exception e) {
            logger.error("服务器错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * App获取网点列表
     *
     * @return
     */
    @ApiOperation(value = "App获取网点列表")
    @GetMapping(value = "/getListOnApp")
    public ResponseEntity<List<BankBranchInfo>> getListOnApp() {
        try {
            QueryWrapper<BankBranchInfo> ew = new QueryWrapper<>();
            ew.ne("id", "0");
            ew.eq("status", 1);
            ew.eq("is_deleted", 0);
            ew.orderByAsc("sortorder");
            ew.orderByDesc("create_time");
            return ResponseEntity.ok(this.bankBranchInfoService.list(ew));
        } catch (Exception e) {
            logger.error("服务器错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    @OperationLog("新建网点")
    @ApiOperation("添加")
    @PostMapping(value = "/add")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/add")
    public ResponseEntity<Map<String, Object>> add(BankBranchInfo bankBranchInfo) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            QueryWrapper<BankBranchInfo> ew = new QueryWrapper<>();
            ew.eq("is_deleted", 0);
            if (StringUtils.isNoneBlank(bankBranchInfo.getName())) {
                ew.eq("name", bankBranchInfo.getName().trim());
            } else {
                resultMap.put("status", 400);
                resultMap.put("message", "请输入网点名称");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            BankBranchInfo one = this.bankBranchInfoService.getOne(ew);
            if (one != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点已存在");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (bankBranchInfo.getSortorder() == null) {
                ew = new QueryWrapper<>();
                ew.select("max(sortorder) as sortorder");
                BankBranchInfo b = this.bankBranchInfoService.getOne(ew);
                if (b == null) {
                    bankBranchInfo.setSortorder(1);
                } else {
                    bankBranchInfo.setSortorder(b.getSortorder() + 1);
                }
            }

            boolean ret = false;
            bankBranchInfo.setCreator(this.getUserId());
            bankBranchInfo.setCreateTime(new Date(System.currentTimeMillis()));
            bankBranchInfo.setUpdateTime(bankBranchInfo.getCreateTime());
            //新建网点时默认在编辑中
            bankBranchInfo.setIsShow(0);
            ret = this.bankBranchInfoService.save(bankBranchInfo);
            if (!ret) {
                resultMap.put("status", 400);
                resultMap.put("message", "添加失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            // 201
            resultMap.put("status", 200);
            resultMap.put("message", "添加成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("添加错误!", e);
        }
        resultMap.put("status", 500);
        resultMap.put("message", "操作失败");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    @OperationLog("修改网点")
    @ApiOperation(value = "编辑")
    @PutMapping(value = "/edit")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/edit")
    public ResponseEntity<Map<String, Object>> edit(BankBranchInfo bankBranchInfo) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            QueryWrapper<BankBranchInfo> ew = new QueryWrapper<>();
            ew.eq("is_deleted", 0);
            if (StringUtils.isNoneBlank(bankBranchInfo.getName())) {
                ew.ne("id", bankBranchInfo.getId());
                ew.eq("name", bankBranchInfo.getName().trim());
            }
            BankBranchInfo one = this.bankBranchInfoService.getOne(ew);
            if (one != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点已存在");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            bankBranchInfo.setModifier(this.getUserId());
            bankBranchInfo.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.bankBranchInfoService.updateById(bankBranchInfo);
            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "修改失败");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            } else {
                resultMap.put("status", 200);
                resultMap.put("message", "修改成功");
                return ResponseEntity.ok(resultMap);
            }
        } catch (Exception e) {
            logger.error("修改错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    /**
     * 刪除
     *
     * @param id
     * @return
     */
    @OperationLog("删除网点")
    @ApiOperation(value = "刪除")
    @DeleteMapping(value = "/delete")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/delete")
    public ResponseEntity<Map<String, Object>> delete(@RequestParam(value = "id", required = true) String id) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            Boolean ret = false;
            //判断该网点下是否有关联用户
            QueryWrapper<Employee> employeeWrapper = new QueryWrapper<>();
            employeeWrapper.eq("bank_branch_id", id);
            employeeWrapper.eq("is_deleted", 0);
            Employee employee = this.employeeService.getOne(employeeWrapper);
            if (employee != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点下存在用户，不允许删除！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            //判断该网点下是否有关联的吐槽建议和呼叫记录
            QueryWrapper<DemandInfo> demandInfoWrapper = new QueryWrapper<>();
            demandInfoWrapper.eq("bank_branch_id", id);
            DemandInfo demandInfo = this.demandInfoService.getOne(demandInfoWrapper);
            if (demandInfo != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点下存在吐槽建议或呼叫记录，不允许删除！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            //判断该网点下是否有关联的业务服务指南
            QueryWrapper<BusinessInfo> businessInfoWrapper = new QueryWrapper<>();
            businessInfoWrapper.eq("bank_branch_id", id);
            BusinessInfo businessInfo = this.businessInfoService.getOne(businessInfoWrapper);
            if (businessInfo != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点下存在业务服务指南，不允许删除！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            //判断该网点下是否有关联的热门产品活动
            QueryWrapper<HotProductActivity> hotProductActivityWrapper = new QueryWrapper<>();
            hotProductActivityWrapper.eq("bank_branch_id", id);
            HotProductActivity hotProductActivity = this.hotProductActivityService.getOne(hotProductActivityWrapper);
            if (hotProductActivity != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点下存在热门产品或活动，不允许删除！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            //判断该网点下是否有关联的热门产品活动预约记录
            QueryWrapper<ProductAppointmentRecord> productAppointmentRecordWrapper = new QueryWrapper<>();
            productAppointmentRecordWrapper.eq("bank_branch_id", id);
            ProductAppointmentRecord productAppointmentRecord = this.productAppointmentRecordService.getOne(productAppointmentRecordWrapper);
            if (productAppointmentRecord != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "该网点下存在热门产品或活动预约记录，不允许删除！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            BankBranchInfo branchInfo = new BankBranchInfo();
            branchInfo.setId(id);
            branchInfo.setIsDeleted(1);
            ret = this.bankBranchInfoService.updateById(branchInfo);

            QueryWrapper<HotSpot> hotSpotWrapper = new QueryWrapper<>();
            hotSpotWrapper.eq("bank_branch_id", id);
            ret = this.hotSpotService.remove(hotSpotWrapper);
            if (!ret) {
                // 删除失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "服务器错误！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 201
            resultMap.put("status", 200);
            resultMap.put("message", "删除成功！");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("服务器错误!", e);
        }
        // 500
        resultMap.put("status", 500);
        resultMap.put("message", "服务器错误！");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    // 批量删除
    @OperationLog("批量删除网点")
    @ApiOperation(value = "批量删除")
    @DeleteMapping(value = "/batchDel")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/batchDel")
    public ResponseEntity<Map<String, Object>> batchDel(String ids) {
        try {
            Map<String, Object> map = new HashMap<>();
            boolean ret = false;
            if (StringUtils.isNotBlank(ids)) {
                String[] idArr = ids.split(",");
                for (String id : idArr) {
                    BankBranchInfo branchInfo = new BankBranchInfo();
                    branchInfo.setId(id);
                    branchInfo.setIsDeleted(1);
                    ret = this.bankBranchInfoService.updateById(branchInfo);
                    if (!ret) {
                        break;
                    }
                }
            } else {
                map.put("status", "400");
                map.put("message", "请选择要删除的网点");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!ret) {
                map.put("status", "400");
                map.put("message", "删除失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            map.put("status", "201");
            map.put("message", "删除成功");
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("服务器错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 获取角色下拉框数据
    @ApiOperation(value = "获取网点下拉框数据")
    @RequestMapping(value = "/getBankBox", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/getBankBox")
    public ResponseEntity<List<BankBranchInfo>> getBankBox(String action, String bankId, String roleId) {
        try {
            QueryWrapper<BankBranchInfo> ew = new QueryWrapper<>();
            ew.select("id", "name");
            ew.eq("is_deleted", 0);
            if (action != null && "create".equals(action) && bankId != null && !"".equals(bankId)) {
                ew.eq("status", 1);
                //新建时归属网点只能选到自己的网点
//                ew.andNew();
//                ew.eq("id", '0');
//                ew.or();
                ew.eq("id", bankId);
            } else if (action != null && "createUser".equals(action)) {
                ew.eq("status", 1);
            } else if (action != null && "appInfoQuery".equals(action)
                    && bankId != null && !"".equals(bankId)
                    && roleId != null && "3".equals(roleId)) {
                //呼叫记录，意见建议，预约记录页面网点归属下拉框，如果是操作员只显示当前网点
                ew.eq("id", bankId);
            } else if (action != null && "query".equals(action)
                    && bankId != null && !"".equals(bankId)
                    && roleId != null && "3".equals(roleId)) {
                //业务，产品，活动页面网点归属下拉框，如果是操作员显示当前网点
                ew.eq("id", bankId);
//                ew.or();
//                ew.eq("id", '0');
            } else if (action != null && "appInfoQuery".equals(action)
                    && roleId != null && !"3".equals(roleId)) {
                //呼叫记录，意见建议页面网点归属下拉框，如果不是操作员显示除了通用的其他网点
                ew.ne("id", '0');
            }
            ew.orderByAsc("sortorder");
            ew.orderByDesc("create_time");
            return ResponseEntity.ok(this.bankBranchInfoService.list(ew));
        } catch (Exception e) {
            logger.error("查询角色列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    @OperationLog("上传网点分布底图")
    @ApiOperation(value = "上传网点分布底图")
    @PutMapping(value = "/uploadImage")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/uploadImage")
    public ResponseEntity<Map<String, Object>> uploadImage(BankBranchInfo bankBranchInfo) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            bankBranchInfo.setModifier(this.getUserId());
            bankBranchInfo.setUpdateTime(new Date(System.currentTimeMillis()));
            bankBranchInfo.setHeadImage(bankBranchInfo.getImageUrl());
            boolean ret = this.bankBranchInfoService.updateById(bankBranchInfo);
            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "上传失败");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            } else {
                resultMap.put("status", 200);
                resultMap.put("message", "上传成功");
                return ResponseEntity.ok(resultMap);
            }
        } catch (Exception e) {
            logger.error("上传错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    @OperationLog("删除网点分布底图")
    @ApiOperation(value = "删除网点分布底图")
    @DeleteMapping(value = "/deleteImage")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/deleteImage")
    public ResponseEntity<Map<String, Object>> deleteImage(BankBranchInfo bankBranchInfo) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            bankBranchInfo.setModifier(this.getUserId());
            bankBranchInfo.setUpdateTime(new Date(System.currentTimeMillis()));
            bankBranchInfo.setImageUrl("");
            bankBranchInfo.setHeadImage(bankBranchInfo.getImageUrl());
            boolean ret = this.bankBranchInfoService.updateById(bankBranchInfo);

            QueryWrapper<HotSpot> hotSpotWrapper = new QueryWrapper<>();
            hotSpotWrapper.eq("bank_branch_id", bankBranchInfo.getId());
            ret = this.hotSpotService.remove(hotSpotWrapper);

            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "删除底图失败");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            } else {
                resultMap.put("status", 200);
                resultMap.put("message", "删除底图成功");
                return ResponseEntity.ok(resultMap);
            }
        } catch (Exception e) {
            logger.error("删除底图错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    @OperationLog("提交网点地图审核")
    @ApiOperation(value = "提交网点地图审核")
    @PutMapping(value = "/submit")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/submit")
    public ResponseEntity<Map<String, Object>> submit(String id) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            BankBranchInfo branchInfo = new BankBranchInfo();
            branchInfo.setId(id);
            branchInfo.setIsShow(2);
            branchInfo.setModifier(this.getUserId());
            branchInfo.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.bankBranchInfoService.updateById(branchInfo);
            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "提交失败");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            } else {
                resultMap.put("status", 200);
                resultMap.put("message", "提交成功");
                return ResponseEntity.ok(resultMap);
            }
        } catch (Exception e) {
            logger.error("提交错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    /**
     * 审核通过网点地图
     *
     * @param id
     * @return
     */
    @OperationLog("审核通过网点地图")
    @ApiOperation(value = "审核通过网点地图")
    @PutMapping(value = "/approved")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/approved")
    public ResponseEntity<Map<String, Object>> approved(String id, Integer status) {
        try {
            Map<String, Object> map = new HashMap<>();
            boolean ret = false;
            if (StringUtils.isNotBlank(id)) {
                BankBranchInfo entity = new BankBranchInfo();
                entity.setId(id);
                if (status == 2) {
                    entity.setIsShow(4);
                } else if (status == 3) {
                    entity.setIsShow(5);
                }
                entity.setReviewer(this.getUserId());
                entity.setUpdateTime(new Date(System.currentTimeMillis()));
                ret = this.bankBranchInfoService.updateById(entity);
            } else {
                map.put("status", "400");
                map.put("message", "请选择需要审核的内容");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!ret) {
                map.put("status", "400");
                map.put("message", "审核通过失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            } else {
                map.put("status", "201");
                map.put("message", "审核通过成功");
                return ResponseEntity.status(HttpStatus.OK).body(map);
            }
        } catch (Exception e) {
            logger.error("审核通过失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    /**
     * 审核驳回
     *
     * @param id
     * @param rejectReason 驳回原因
     * @return
     */
    @OperationLog("审核驳回产品或活动")
    @ApiOperation(value = "审核驳回")
    @PutMapping(value = "/reject")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/reject")
    public ResponseEntity<Map<String, Object>> reject(String id, String rejectReason, Integer status) {
        try {
            Map<String, Object> map = new HashMap<>();
            if (rejectReason == null || "".equals(rejectReason.trim())) {
                map.put("status", "400");
                map.put("message", "请输入驳回原因！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }

            boolean ret = false;
            if (StringUtils.isNotBlank(id)) {
                BankBranchInfo entity = new BankBranchInfo();
                entity.setId(id);
                if (status != null) {
                    if (status == 2) {
                        entity.setIsShow(1);
                    } else if (status == 3) {
                        entity.setIsShow(6);
                    }
                }
                entity.setRejectReason(rejectReason);
                entity.setReviewer(this.getUserId());
                entity.setUpdateTime(new Date(System.currentTimeMillis()));
                ret = this.bankBranchInfoService.updateById(entity);
            } else {
                map.put("status", "400");
                map.put("message", "请选择需要审核的内容");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!ret) {
                map.put("status", "400");
                map.put("message", "审核驳回失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            } else {
                map.put("status", "201");
                map.put("message", "审核驳回成功");
                return ResponseEntity.status(HttpStatus.OK).body(map);
            }
        } catch (Exception e) {
            logger.error("审核驳回失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    /**
     * 申请下线
     *
     * @param id
     * @return
     */
    @OperationLog("申请下线单个网点地图")
    @ApiOperation(value = "申请下线单个网点地图")
    @PutMapping(value = "/applyOffline")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/applyOffline")
    public ResponseEntity<Map<String, Object>> applyOffline(String id) {
        try {
            Map<String, Object> map = new HashMap<>();
            BankBranchInfo entity = new BankBranchInfo();
            entity.setId(id);
            entity.setIsShow(3);
            entity.setModifier(this.getUserId());
            entity.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.bankBranchInfoService.updateById(entity);
            if (!ret) {
                map.put("status", "400");
                map.put("message", "申请下线错误");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            map.put("status", "201");
            map.put("message", "申请下线成功");
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("申请下线出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(
                null);
    }

    /**
     * 申请上线
     *
     * @param id
     * @return
     */
    @OperationLog("申请上线单个网点地图")
    @ApiOperation(value = "申请上线单个网点地图")
    @PutMapping(value = "/applyOnline")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/applyOnline")
    public ResponseEntity<Map<String, Object>> applyOnline(String id) {
        try {
            Map<String, Object> map = new HashMap<>();
            BankBranchInfo entity = new BankBranchInfo();
            entity.setId(id);
            entity.setIsShow(2);
            entity.setModifier(this.getUserId());
            entity.setUpdateTime(new Date(System.currentTimeMillis()));
            boolean ret = this.bankBranchInfoService.updateById(entity);
            if (!ret) {
                map.put("status", "500");
                map.put("message", "服务器错误");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(map);
            }
            map.put("status", "201");
            map.put("message", "申请上线成功");
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("申请上线出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(
                null);
    }

    /**
     * 网点排序（供APP端展示下拉框内容使用）
     *
     * @param oldList 前端排序后的List
     * @return
     */
    @OperationLog("网点排序")
    @ApiOperation(value = "网点排序")
    @PutMapping(value = "/sort")
    @RequiresAuthentication  //@RequiresPermissions("/bankBranchInfo/sort")
    public ResponseEntity<Map<String, Object>> sort(List<BankBranchInfo> oldList) {
        try {
            Map<String, Object> map = new HashMap<>();
            // 对顺序号赋值封装
            List<BankBranchInfo> newList = new ArrayList<BankBranchInfo>();
            boolean ret = false;
            if (oldList.size() > 0) {
                //先清空之前的排序
                QueryWrapper<BankBranchInfo> ew = new QueryWrapper<>();
                ew.ne("id", "0");
                ew.isNull("sortorder");
                ret = this.bankBranchInfoService.update(ew);

                //重新排序
                for (int i = 0; i < oldList.size(); i++) {
                    BankBranchInfo entity = oldList.get(i);
                    entity.setSortorder(i + 1);
                    newList.add(entity);
                }
                ret = this.bankBranchInfoService.updateBatchById(newList);
            } else {
                map.put("status", "400");
                map.put("message", "请选择需要排序的网点");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!ret) {
                map.put("status", "400");
                map.put("message", "排序失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            } else {
                map.put("status", "201");
                map.put("message", "排序成功");
                return ResponseEntity.status(HttpStatus.OK).body(map);
            }
        } catch (Exception e) {
            logger.error("排序失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

}

