package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.model.DemandInfo;
import cn.wisenergy.chnmuseum.party.model.Employee;
import cn.wisenergy.chnmuseum.party.service.impl.DemandInfoServiceImpl;
import cn.wisenergy.chnmuseum.party.service.impl.EmployeeServiceImpl;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.vdurmont.emoji.EmojiParser;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 需求建议信息 前端控制器
 * </p>
 *
 * @author 杨智平
 * @since 2018-08-29
 */
@Api(tags = "呼叫服务或者吐槽建议")
@RestController
@RequestMapping("/demandInfo")
public class DemandInfoController extends BaseController {

    private static final Logger logger = LoggerFactory.getLogger(EmployeeController.class);

    @Resource
    private DemandInfoServiceImpl demandInfoService;

    @Resource
    private EmployeeServiceImpl employeeService;

    /**
     * 获取单个呼叫服务或者吐槽建议
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "获取单个呼叫服务或者吐槽建议")
    @GetMapping(value = "/getById")
    @RequiresAuthentication  //@RequiresPermissions("/demandInfo/getById")
    public ResponseEntity<DemandInfo> getById(String id) {
        try {
            DemandInfo one = this.demandInfoService.selectOneById(id);
            if (null != one) {
                one.setContent(EmojiParser.parseToUnicode(one.getContent()));
                if (StringUtils.isNotBlank(one.getFeedbackInfo())) {
                    one.setFeedbackInfo(EmojiParser.parseToUnicode(one.getFeedbackInfo()));
                }
                return ResponseEntity.ok(one);
            }
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
        } catch (Exception e) {
            logger.error("获取单个呼叫服务或者吐槽建议出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }


    /**
     * 获取呼叫服务或者吐槽建议列表
     *
     * @param lineNumber
     * @param content
     * @param phoneNumber
     * @param type
     * @param time
     * @return
     */
    @ApiOperation(value = "获取呼叫服务或者吐槽建议列表")
    @GetMapping(value = "/getList")
    @RequiresAuthentication  //@RequiresPermissions("/demandInfo/getList")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "lineNumber", value = "叫号单号", required = false),
            @ApiImplicitParam(name = "content", value = "需求内容", required = false, dataType = "String"),
            @ApiImplicitParam(name = "phoneNumber", value = "手机号码", required = false, dataType = "String"),
            @ApiImplicitParam(name = "type", value = "消息类型：1-吐槽建议2-呼叫记录", required = true),
            @ApiImplicitParam(name = "time", value = "反馈日期", required = false, dataType = "String"),
            @ApiImplicitParam(name = "callTime", value = "呼叫日期", required = false, dataType = "String"),
            @ApiImplicitParam(name = "bankBranchId", value = "归属网点ID", required = false, dataType = "String")})
    public ResponseEntity<Page<DemandInfo>> getList(String lineNumber, String content, String phoneNumber,
                                                    Integer type, String time, String callTime, String bankBranchId,
                                                    String roleId, String currentBankId) {
        try {
            if (!"3".equals(roleId)) {
                roleId = "0";
            }
            String startDate = null;
            String endDate = null;
            String callStartDate = null;
            String callEndDate = null;
            if (StringUtils.isNoneBlank(time)) {
                startDate = time + " 00:00:00";
                endDate = time + " 23:59:59";
            }
            if (StringUtils.isNoneBlank(callTime)) {
                callStartDate = callTime + " 00:00:00";
                callEndDate = callTime + " 23:59:59";
            }
            lineNumber = StringUtils.trimToNull(lineNumber); // 清除掉str首尾的空白字符,如果仅str全由空白字符组成则返回null
            content = StringUtils.trimToNull(content);
            phoneNumber = StringUtils.trimToNull(phoneNumber);
            Page<DemandInfo> page = this.getPage();
            Page<DemandInfo> demandInfoPage = this.demandInfoService.selectDemandInfoList(page, lineNumber, content, phoneNumber,
                    startDate, endDate, type, bankBranchId, callStartDate, callEndDate, roleId, currentBankId);
            for (DemandInfo demandInfo : demandInfoPage.getRecords()) {
                demandInfo.setContent(EmojiParser.parseToUnicode(demandInfo.getContent()));
                if (StringUtils.isNotBlank(demandInfo.getFeedbackInfo())) {
                    demandInfo.setFeedbackInfo(EmojiParser.parseToUnicode(demandInfo.getFeedbackInfo()));
                }
            }
            return ResponseEntity.ok(demandInfoPage);
        } catch (Exception e) {
            logger.error("服务器错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 添加
     */
    @Transactional
    @ApiOperation("APP端添加吐槽建议或者呼叫记录")
    @PostMapping(value = "/add")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "lineNumber", value = "叫号单号", required = true),
            @ApiImplicitParam(name = "content", value = "需求内容", required = true, dataType = "String"),
            @ApiImplicitParam(name = "phoneNumber", value = "手机号码", required = true, dataType = "String"),
            @ApiImplicitParam(name = "type", value = "消息类型：1-吐槽建议2-呼叫记录", required = true)})
    public ResponseEntity<Map<String, Object>> add(DemandInfo demandInfo) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            boolean ret = false;
            //接收人员工号，多人使用|分割，如"01014760|01158451"
            String recUsrNbr = "";
            //反馈人ID
            String personId = "";
            if (demandInfo.getBankBranchId() != null) {
                Employee emp = this.employeeService.getEmpCodeBybankId(demandInfo.getBankBranchId());
                if (emp != null) {
                    recUsrNbr = emp.getCode();
                    personId = emp.getId();
                    //现在改为反馈的时候再存入反馈人ID
//                    demandInfo.setFeedbackPersonId(emp.getId());
                }
            }
            demandInfo.setLineNumber(StringUtils.trimToNull(demandInfo.getLineNumber()));// 清除掉str首尾的空白字符,如果仅str全由空白字符组成则返回null
            demandInfo.setContent(EmojiParser.parseToHtmlHexadecimal(demandInfo.getContent().trim()));
            demandInfo.setPhoneNumber(StringUtils.trimToNull(demandInfo.getPhoneNumber()));
            demandInfo.setCreateTime(new Date(System.currentTimeMillis()));
            ret = this.demandInfoService.save(demandInfo);
            if (!ret) {
                resultMap.put("status", 400);
                resultMap.put("message", "提交失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            if (!"".equals(recUsrNbr)) {
                //招乎正文，要求以[发送人姓名OR发送人部室]正文格式
                String zhCnt = "";
                if (demandInfo.getType() == 1) {
                    //吐槽建议
                    zhCnt = "客户" + demandInfo.getLineNumber() + "号（"
                            + demandInfo.getPhoneNumber() + "）有吐槽建议，请及时反馈改进，"
                            + "并点击链接" + demandInfo.getId() + "&personId=" + personId + "及时回复客户。";
                } else if (demandInfo.getType() == 2) {
                    //呼叫大堂
                    zhCnt = "客户" + demandInfo.getLineNumber() + "号（"
                            + demandInfo.getPhoneNumber() + "）呼叫大堂，内容为'"
                            + demandInfo.getContent() + "'，请及时服务接待！";

                }
                logger.info(zhCnt);
            }
            // 201
            resultMap.put("status", 200);
            resultMap.put("message", "提交成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("提交错误!", e);
        }
        resultMap.put("status", 500);
        resultMap.put("message", "操作失败");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    @ApiOperation(value = "填写反馈内容")
    @PutMapping(value = "/edit")
    public ResponseEntity<Map<String, Object>> edit(String id, String feedbackInfo, String personId) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            if (!StringUtils.isNoneBlank(id)) {
                resultMap.put("status", 400);
                resultMap.put("message", "页面信息失效，请重新点击链接进入");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            //验证反馈人是否存在
            Employee employee = employeeService.getById(personId);
            if (employee == null) {
                resultMap.put("status", 400);
                resultMap.put("message", "反馈链接有误");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (!StringUtils.isNoneBlank(feedbackInfo)) {
                resultMap.put("status", 400);
                resultMap.put("message", "请输入反馈内容");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            } else {
                feedbackInfo = StringUtils.trimToNull(feedbackInfo); // 清除掉str首尾的空白字符,如果仅str全由空白字符组成则返回null
            }

            DemandInfo entity = this.demandInfoService.getById(id);
            if (entity.getFeedbackInfo() != null) {
                resultMap.put("status", 400);
                resultMap.put("message", "您已反馈，请勿重复提交！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            //保存反馈人ID
            entity.setFeedbackPersonId(personId);
            entity.setFeedbackTime(new Date(System.currentTimeMillis()));
            entity.setFeedbackInfo(EmojiParser.parseToHtmlHexadecimal(feedbackInfo.trim()));
            boolean ret = this.demandInfoService.updateById(entity);
            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "反馈失败");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            } else {
                resultMap.put("status", 200);
                resultMap.put("message", "反馈成功");
                return ResponseEntity.ok(resultMap);
            }
        } catch (Exception e) {
            logger.error("反馈错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    /**
     * App获取呼叫服务或者吐槽建议列表
     *
     * @param type
     * @param advisorId
     * @param bankBranchId
     * @return
     */
    @ApiOperation(value = "App获取呼叫服务或者吐槽建议列表")
    @GetMapping(value = "/getListOnApp")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "type", value = "消息类型：1-吐槽建议2-呼叫记录", required = true),
            @ApiImplicitParam(name = "advisorId", value = "发送人ID(当前登陆用户Id)", required = true, dataType = "String"),
            @ApiImplicitParam(name = "bankBranchId", value = "归属网点ID", required = true, dataType = "String")})
    public ResponseEntity<List<DemandInfo>> getListOnApp(Integer type, String advisorId, String bankBranchId) {
        try {

            QueryWrapper<DemandInfo> demandInfoWrapper = new QueryWrapper<>();
            demandInfoWrapper.eq("type", type)
                    .eq("advisor_id", advisorId)
                    .eq("bank_branch_id", bankBranchId)
                    .orderByDesc("create_time");
            List<DemandInfo> demandInfoList = this.demandInfoService.list(demandInfoWrapper);
            for (DemandInfo demandInfo : demandInfoList) {
                demandInfo.setContent(EmojiParser.parseToUnicode(demandInfo.getContent()));
                if (StringUtils.isNotBlank(demandInfo.getFeedbackInfo())) {
                    demandInfo.setFeedbackInfo(EmojiParser.parseToUnicode(demandInfo.getFeedbackInfo()));
                }
            }
            return ResponseEntity.ok(demandInfoList);
        } catch (Exception e) {
            logger.error("服务器错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * APP获取单个呼叫服务或者吐槽建议
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "App获取单个呼叫服务或者吐槽建议")
    @GetMapping(value = "/getByIdForApp")
    public ResponseEntity<DemandInfo> getByIdForApp(String id) {
        try {
            QueryWrapper<DemandInfo> ew = new QueryWrapper<>();
            ew.select("id,line_number AS lineNumber,create_time AS createTime,phone_number AS phoneNumber,content,feedback_info AS feedbackInfo");
            ew.eq("id", id);
            DemandInfo one = this.demandInfoService.getOne(ew);
            if (null != one) {
                return ResponseEntity.ok(one);
            }
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
        } catch (Exception e) {
            logger.error("获取单个呼叫服务或者吐槽建议出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }



}

