package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.auth.SHA256PasswordEncryptionService;
import cn.wisenergy.chnmuseum.party.auth.SecureRandomSaltService;
import cn.wisenergy.chnmuseum.party.common.util.RandomUtil;
import cn.wisenergy.chnmuseum.party.core.annotations.OperationLog;
import cn.wisenergy.chnmuseum.party.model.BankBranchInfo;
import cn.wisenergy.chnmuseum.party.model.Employee;
import cn.wisenergy.chnmuseum.party.model.EmployeeRole;
import cn.wisenergy.chnmuseum.party.service.RoleService;
import cn.wisenergy.chnmuseum.party.service.impl.BankBranchInfoServiceImpl;
import cn.wisenergy.chnmuseum.party.service.impl.EmployeeRoleServiceImpl;
import cn.wisenergy.chnmuseum.party.service.impl.EmployeeServiceImpl;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * <p>
 * 组织成员 前端控制器
 * </p>
 *
 * @author 杨智平
 * @since 2018-08-02
 */
@Api(tags = "成员相关接口")
@RestController
@RequestMapping("/employee")
public class EmployeeController extends BaseController {

    private static final Logger logger = LoggerFactory.getLogger(EmployeeController.class);

    @Resource
    private EmployeeServiceImpl employeeService;

    @Resource
    private EmployeeRoleServiceImpl employeeRoleService;

    @Resource
    private StringRedisTemplate stringRedisTemplate;
    private static final String SHIRO_JWT_TOKEN = "shiro:jwt:token";
    //用户登录是否被锁定    一小时 redisKey 前缀
    private String SHIRO_IS_LOCK = "shiro_is_lock_";

    @Resource
    private BankBranchInfoServiceImpl bankBranchInfoService;

    @Resource
    private RoleService roleService;

    /**
     * 获取单个成员信息
     *
     * @param Id
     * @return
     */
    @ApiOperation(value = "获取单个成员信息")
    @GetMapping(value = "/get/")
    @RequiresAuthentication  //@RequiresPermissions("/employee/get/")
    public ResponseEntity<Employee> getById(String Id) {
        try {
            Employee employee = employeeService.selectByEmpId(Id);
            BankBranchInfo bankBranch = this.bankBranchInfoService.getById(employee.getBankBranchId());
            if (bankBranch != null) {
                employee.setBankBranchName(bankBranch.getName());
            }
            if (null == employee) {
                return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
            }
            return ResponseEntity.ok(employee);
        } catch (Exception e) {
            logger.error("查询成员信息错误！", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 查询成员列表
     *
     * @param employName (用户名称，账号，手机号，角色名称有一个满足即可返回值)
     * @return
     */
    @ApiOperation(value = "查询成员列表")
    @RequestMapping(value = "/getUserList", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/employee/getUserList")
    public ResponseEntity<Page<Employee>> queryUserList(String employName) {
        try {
            employName = StringUtils.trimToNull(employName);
            Page<Employee> page = getPage();
            Page<Employee> employeePage = employeeService.selectRolenameList(page, employName);
            return ResponseEntity.ok(employeePage);
        } catch (Exception e) {
            logger.error("查询成员列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    //新增
    @OperationLog("新增成员")
    @ApiOperation(value = "新增成员")
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    @RequiresAuthentication  //@RequiresPermissions("/employee/add")
    public ResponseEntity<Map<String, Object>> add(Employee employee) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            if (StringUtils.isBlank(employee.getUsername())) {
                resultMap.put("status", 400);
                resultMap.put("message", "账号不能为空！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            } else {
                employee.setUsername(StringUtils.trimToNull(employee.getUsername()));
            }
            if (StringUtils.isBlank(employee.getPassword())) {
                resultMap.put("status", 400);
                resultMap.put("message", "密码不能为空！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            } else {
                employee.setPassword(StringUtils.trimToNull(employee.getPassword()));
            }
            if (StringUtils.isBlank(employee.getName())) {
                resultMap.put("status", 400);
                resultMap.put("message", "姓名不能为空！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            } else {
                employee.setName(StringUtils.trimToNull(employee.getName()));
            }
            if (StringUtils.isBlank(employee.getCode())) {
                resultMap.put("status", 400);
                resultMap.put("message", "员工号不能为空！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            } else {
                employee.setCode(StringUtils.trimToNull(employee.getCode()));
            }
            if (StringUtils.isBlank(employee.getRoleId())) {
                resultMap.put("status", 400);
                resultMap.put("message", "请选择角色！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (StringUtils.isBlank(employee.getBankBranchId())) {
                resultMap.put("status", 400);
                resultMap.put("message", "请选择归属网点！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            QueryWrapper<Employee> ew = new QueryWrapper<>();
            if (StringUtils.isNoneBlank(employee.getUsername())) {
                employee.setUsername(employee.getUsername().trim());
                ew.eq("is_deleted", 0);
                ew.eq("username", employee.getUsername());
                Employee one = this.employeeService.getOne(ew);
                if (one != null) {
                    resultMap.put("status", 400);
                    resultMap.put("message", "账号已存在！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                }
            }
            /*if (StringUtils.isNoneBlank(employee.getCode())) {
                employee.setCode(employee.getCode().trim());
                ew = new QueryWrapper<>();
                ew.eq("is_deleted", 0);
                ew.eq("code", employee.getCode());
                Employee one = this.employeeService.getOne(ew);
                if (one != null) {
                    resultMap.put("status", 400);
                    resultMap.put("message", "该员工号已存在！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                }
            }*/
            //判断是否存在已启用的大堂主管
            if ("4".equals(employee.getRoleId())) {
                Boolean hasEnHallDirector = this.employeeService.isHasEnHallDirector(null, employee.getBankBranchId());
                if (hasEnHallDirector) {
                    resultMap.put("status", 400);
                    resultMap.put("message", "该网点已存在启用的大堂主管！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                }
            }

            boolean ret = false;
            if (employee.getSortorder() == null) {
                ew = new QueryWrapper<>();
                ew.select("max(sortorder) as sortorder");
                Employee e = this.employeeService.getOne(ew);
                if (e == null) {
                    employee.setSortorder(1);
                } else {
                    employee.setSortorder(e.getSortorder() + 1);
                }
            }
            byte[] passwordSalt = SecureRandomSaltService.generateSalt();
            byte[] passwordHash = SHA256PasswordEncryptionService
                    .createPasswordHash(employee.getPassword(), passwordSalt);
            employee.setPasswordSalt(passwordSalt);
            employee.setPasswordHash(passwordHash);
            employee.setCreateTime(new Date(System.currentTimeMillis()));
            employee.setUpdateTime(employee.getCreateTime());
            employee.setSex(employee.getSex());
            ret = this.employeeService.save(employee);

            EmployeeRole entity = new EmployeeRole();
            entity.setEmployeeId(employee.getId());
            entity.setRoleId(employee.getRoleId());
            entity.setCreateTime(new Date(System.currentTimeMillis()));
            entity.setUpdateTime(entity.getCreateTime());
            this.employeeRoleService.save(entity);

            if (!ret) {
                // 新增失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "服务器忙");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR)
                        .body(resultMap);
            }
            resultMap.put("status", 200);
            resultMap.put("message", "添加成功");
            // 201
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            resultMap.put("status", 500);
            resultMap.put("message", "服务器忙");
            logger.error("新增成员错误!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }


    // 编辑用户信息
    @OperationLog("修改成员信息")
    @ApiOperation(value = "编辑用户信息(必须传 1username 2name 3roleId)")
    @PutMapping(value = "/modify")
    @RequiresAuthentication  //@RequiresPermissions("/employee/modify")
    public ResponseEntity<Map<String, Object>> edit(Employee employee) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean ret = false;
            if (employee.getId() != null) {
                if ("1".equals(employee.getId()) && employee.getStatus() == false) {
                    resultMap.put("status", 400);
                    resultMap.put("message", "该账号不能被禁用");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                }
                //判断是否存在已启用的大堂主管
                if ("4".equals(employee.getRoleId()) && employee.getStatus()) {
                    Boolean hasEnHallDirector = this.employeeService.isHasEnHallDirector(employee.getId(), employee.getBankBranchId());
                    if (hasEnHallDirector) {
                        resultMap.put("status", 400);
                        resultMap.put("message", "该网点已存在启用的大堂主管！");
                        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                    }
                }

                employee.setUsername(StringUtils.trimToNull(employee.getUsername()));
                employee.setPassword(StringUtils.trimToNull(employee.getPassword()));
                employee.setName(StringUtils.trimToNull(employee.getName()));
                employee.setUpdateTime(new Date(System.currentTimeMillis()));
                ret = employeeService.updateById(employee);
                //查询当前用户拥有的角色
                QueryWrapper<EmployeeRole> employeeRoleWrapper = new QueryWrapper<>();
                employeeRoleWrapper.eq("employee_id", employee.getId());
                EmployeeRole employeeRole = this.employeeRoleService.getOne(employeeRoleWrapper);

                if (employeeRole != null && employee.getRoleId() != null
                        && employee.getRoleId() != employeeRole.getRoleId()) {
                    // 角色有变化即修改角色
                    employeeRole.setRoleId(employee.getRoleId());
                    employeeRole.setUpdateTime(employee.getUpdateTime());
                    ret = this.employeeRoleService.updateById(employeeRole);
                }
            } else {
                // 更新失败, 400
                resultMap.put("status", 400);
                resultMap.put("message", "请选择用户");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            if (!ret) {
                // 更新失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "服务器忙");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "更新成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("更新错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    /**
     * 删除
     *
     * @param employeeId
     * @return
     */
    @OperationLog("删除成员")
    @ApiOperation(value = "删除成员")
    @DeleteMapping(value = "/delete")
    @RequiresAuthentication  //@RequiresPermissions("/employee/delete")
    public ResponseEntity<Map<String, Object>> delete(String employeeId) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            if ("1".equals(employeeId)) {
                resultMap.put("status", 400);
                resultMap.put("message", "该账号不能被删除");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            Employee entity = new Employee();
            entity.setId(employeeId);
            entity.setUpdateTime(new Date(System.currentTimeMillis()));
            entity.setIsDeleted(1);
            boolean ret1 = this.employeeService.updateById(entity);

            QueryWrapper<EmployeeRole> employeeRoleWrapper = new QueryWrapper<>();
            employeeRoleWrapper.eq("employee_id", employeeId);
            boolean ret2 = this.employeeRoleService.remove(employeeRoleWrapper);

            if (!ret1 || !ret2) {
                resultMap.put("status", 400);
                resultMap.put("message", "删除失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            resultMap.put("status", 200);
            resultMap.put("message", "删除成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("删除用户出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    // 批量删除
    @OperationLog("批量删除成员")
    @ApiOperation(value = "批量删除")
    @DeleteMapping(value = "/batchDel")
    @RequiresAuthentication  //@RequiresPermissions("/employee/batchDel")
    public ResponseEntity<Map<String, Object>> batchDel(String employeeIds) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean flag = false;
            boolean batchDelflag = false;
            String empId = "";
            if (StringUtils.isNotBlank(employeeIds)) {
                String[] employeeIdArr = employeeIds.split(",");
                Employee entity = new Employee();
                entity.setIsDeleted(1);
                for (String employeeId : employeeIdArr) {
                    if ("1".equals(employeeId)) {
                        empId = "1";
                        continue;
                    }
                    entity.setId(employeeId);
                    flag = this.employeeService.updateById(entity);

                    QueryWrapper<EmployeeRole> employeeRoleWrapper = new QueryWrapper<>();
                    employeeRoleWrapper.eq("employee_id", employeeId);
                    flag = this.employeeRoleService.remove(employeeRoleWrapper);
                    if (!flag) {
                        batchDelflag = true;
                        break;
                    }
                }
            } else {
                resultMap.put("status", 400);
                resultMap.put("message", "删除失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (batchDelflag) {
                resultMap.put("status", 400);
                resultMap.put("message", "删除失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (!"".equals(empId) && !flag) {
                resultMap.put("status", 400);
                resultMap.put("message", "系统用户不能被删除");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            resultMap.put("status", 200);
            String message = "删除成功";
            if (!"".equals(empId) && flag) {
                message = "admin不能被删除，其他选中用户删除成功";
            }
            resultMap.put("message", message);
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("删除失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    /**
     * 禁用
     *
     * @param employeeId
     * @return
     */
    @OperationLog("禁用成员")
    @ApiOperation(value = "禁用")
    @RequestMapping(value = "/disable", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/employee/disable")
    public ResponseEntity<Map<String, Object>> disableEmployee(String employeeId) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            Employee entity = new Employee();
            entity.setId(employeeId);
            entity.setStatus(false);
            boolean ret = this.employeeService.updateById(entity);

            //获取该用户的登陆token
            String userToken = stringRedisTemplate.opsForValue().get(SHIRO_JWT_TOKEN + employeeId);
            if (null != userToken) {
                stringRedisTemplate.delete(userToken);
                stringRedisTemplate.delete(SHIRO_JWT_TOKEN + employeeId);
            }
            if (!ret) {
                resultMap.put("status", 400);
                resultMap.put("message", "禁用失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            resultMap.put("status", 200);
            resultMap.put("message", "禁用成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("禁用用户出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    // 批量禁用
    @OperationLog("批量禁用成员")
    @ApiOperation(value = "批量禁用")
    @RequestMapping(value = "/batchDis", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/employee/batchDis")
    public ResponseEntity<Map<String, Object>> batchDis(String employeeIds) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean flag = false;
            if (StringUtils.isNotBlank(employeeIds)) {
                String[] employeeIdArr = employeeIds.split(",");
                Employee entity = new Employee();
                entity.setStatus(false);
                for (String employeeId : employeeIdArr) {
                    if ("1".equals(employeeId)) {
                        continue;
                    }
                    entity.setId(employeeId);
                    flag = this.employeeService.updateById(entity);
                    //获取该用户的登陆token
                    String userToken = stringRedisTemplate.opsForValue().get(SHIRO_JWT_TOKEN + employeeId);
                    if (null != userToken) {
                        stringRedisTemplate.delete(userToken);
                        stringRedisTemplate.delete(SHIRO_JWT_TOKEN + employeeId);
                    }
                    if (!flag) {
                        break;
                    }
                }
            } else {
                resultMap.put("status", 400);
                resultMap.put("message", "禁用失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (!flag) {
                resultMap.put("status", 400);
                resultMap.put("message", "禁用失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            resultMap.put("status", 200);
            resultMap.put("message", "禁用成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("用户禁用失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 启动
    @OperationLog("启用成员")
    @ApiOperation(value = "启用")
    @RequestMapping(value = "/enable", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/employee/enable")
    public ResponseEntity<Map<String, Object>> enableUser(String employeeId, String currBankId) {
        try {
            Map<String, Object> map = new HashMap<>();

            //判断该网点下是否有启用的大堂主管
            Boolean hasEnHallDirector = this.employeeService.isHasEnHallDirector(employeeId, currBankId);
            if (hasEnHallDirector) {
                map.put("status", "400");
                map.put("message", "该网点存在启用的大堂主管，如需启用，请先禁用其他大堂主管");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }

            Employee entity = new Employee();
            entity.setId(employeeId);
            entity.setStatus(true);
            boolean ret = this.employeeService.updateById(entity);
            if (!ret) {
                map.put("status", "500");
                map.put("message", "服务器错误");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(map);
            }
            map.put("status", "201");
            map.put("message", "启用成功");
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("用户启用出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(
                null);
    }

    // 批量启动
    @OperationLog("批量启用成员")
    @ApiOperation(value = "批量启动")
    @RequestMapping(value = "/batchEnable", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/employee/batchEnable")
    public ResponseEntity<Map<String, Object>> batchEnable(String employeeIds) {
        try {
            Map<String, Object> map = new HashMap<>();
            boolean flag = false;
            if (StringUtils.isNotBlank(employeeIds)) {
                String[] employeeIdArr = employeeIds.split(",");
                Employee entity = new Employee();
                entity.setStatus(true);
                for (String employeeId : employeeIdArr) {
                    entity.setId(employeeId);
                    flag = this.employeeService.updateById(entity);
                    if (!flag) {
                        break;
                    }
                }
            } else {
                map.put("status", "400");
                map.put("message", "请选择要启动的用户");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            }
            if (!flag) {
                map.put("status", "400");
                map.put("message", "批量启动失败");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(map);
            } else {
                map.put("status", "201");
                map.put("message", "批量启动成功");
                return ResponseEntity.status(HttpStatus.OK).body(map);
            }
        } catch (Exception e) {
            logger.error("批量启动失败!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    @OperationLog("修改密码")
    @ApiOperation(value = "管理员更改自己的登录密码", notes = "管理员更改自己的登录密码")
    @RequestMapping(value = "/editPwd", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/employee/editPwd")
    public ResponseEntity<Map<String, Object>> editPwd(@RequestParam(value = "oldPassWord", required = true) String oldPassWord,
                                                       @RequestParam(value = "password", required = true) String password) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            boolean ret = false;
            Employee employee = this.employeeService.getById(this.getUserId());
            byte[] salt = employee.getPasswordSalt();
            if (new String(SHA256PasswordEncryptionService.createPasswordHash(oldPassWord, salt))
                    .equals(new String(employee.getPasswordHash()))) {
                salt = SecureRandomSaltService.generateSalt();
                employee.setPasswordSalt(salt);
                employee.setPasswordHash((SHA256PasswordEncryptionService.createPasswordHash(password, salt)));
                employee.setUpdateTime(new Date(System.currentTimeMillis()));
                ret = this.employeeService.updateById(employee);
            } else {
                logger.error("旧密码不正确");
                resultMap.put("status", 400);
                resultMap.put("message", "旧密码不正确");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (!ret) {
                resultMap.put("status", 500);
                resultMap.put("message", "修改失败");
                // 更新失败, 500
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            resultMap.put("status", 200);
            resultMap.put("message", "修改成功");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("更新密码错误！", e);
        }
        // 500
        resultMap.put("status", 500);
        resultMap.put("message", "修改失败");
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
    }

    // 管理员重置密码
    @OperationLog("重置密码")
    @ApiOperation(value = "管理员重置密码", notes = "管理员重置密码")
    @RequestMapping(value = "/resetPassword", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/employee/resetPassword")
    public ResponseEntity<Map<Object, String>> resetPassword(String employeeId) {
        try {
            Map<Object, String> map = new LinkedHashMap<>();
            Employee employee = new Employee();
            employee.setId(employeeId);
            String newPassword = RandomUtil.createRandom(6);
            employee.setPassword(newPassword);
            byte[] passwordSalt = SecureRandomSaltService.generateSalt();
            byte[] passwordHash = SHA256PasswordEncryptionService.createPasswordHash(employee.getPassword(), passwordSalt);
            employee.setPasswordSalt(passwordSalt);
            employee.setPasswordHash(passwordHash);
            boolean ret = this.employeeService.updateById(employee);
            if (!ret) {
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
            }
            stringRedisTemplate.delete(SHIRO_IS_LOCK + this.employeeService.selectByEmpId(employeeId).getUsername());
            map.put("status", "201");
            map.put("message", "重置密码成功");
            map.put("password", newPassword);
            return ResponseEntity.status(HttpStatus.OK).body(map);
        } catch (Exception e) {
            logger.error("重置密码出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 给支行管理员查询自己网点人员
     * 2019-01-24增加新的模块-yjl
     *
     * @param employName (用户名称，账号，手机号，角色名称有一个满足即可返回值)
     * @return
     */
    @ApiOperation(value = "给支行管理员查询自己网点人员")
    @RequestMapping(value = "/selectOwnEmpList", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/employee/selectOwnEmpList")
    public ResponseEntity<Page<Employee>> selectOwnEmpList(String employName, String currBankID) {
        try {
            employName = StringUtils.trimToNull(employName);
            Page<Employee> page = getPage();
            Page<Employee> employeePage = employeeService.selectOwnEmpList(page, employName, currBankID);
            return ResponseEntity.ok(employeePage);
        } catch (Exception e) {
            logger.error("查询成员列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }
}

