package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.common.dfs.FastDFSUtils;
import cn.wisenergy.chnmuseum.party.common.enums.*;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Add;
import cn.wisenergy.chnmuseum.party.common.validator.groups.Update;
import cn.wisenergy.chnmuseum.party.common.vo.GenericPageParam;
import cn.wisenergy.chnmuseum.party.model.Asset;
import cn.wisenergy.chnmuseum.party.model.Audit;
import cn.wisenergy.chnmuseum.party.model.ExhibitionBoard;
import cn.wisenergy.chnmuseum.party.model.VideoContent;
import cn.wisenergy.chnmuseum.party.service.*;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <pre>
 * 展板 前端控制器
 * </pre>
 *
 * @author Danny Lee
 * @since 2021-03-19
 */
@Slf4j
@RestController
@RequestMapping("/exhibitionBoard")
@Api(tags = {"展板操作接口"})
public class ExhibitionBoardController extends BaseController {

    @Resource
    private ExhibitionBoardService exhibitionBoardService;
    @Resource
    private ExhibitionBoardCatService exhibitionBoardCatService;
    @Resource
    private CopyrightOwnerService copyrightOwnerService;
    @Resource
    private VideoContentService videoContentService;
    @Resource
    private AuditService auditService;
    @Resource
    private AssetService assetService;

    @PostMapping("/save")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:save")
    @ApiOperation(value = "添加展板", notes = "添加展板")
    public Map<String, Object> saveExhibitionBoard(@Validated(value = {Add.class}) ExhibitionBoard exhibitionBoard) {
        final List<String> audioIdList = exhibitionBoard.getAudioIdList();
        if (audioIdList == null || audioIdList.isEmpty()) {
            return getFailResult("400", "导览音频文件必须上传");
        }
        final List<String> datumIdList = exhibitionBoard.getDatumIdList();
        if (datumIdList == null || datumIdList.isEmpty()) {
            return getFailResult("400", "参考资料文件必须上传");
        }
        exhibitionBoard.setAuditStatus(AuditStatusEnum.TBC.name());
        exhibitionBoard.setPublished(false);
        exhibitionBoard.setDeleted(false);
        // 保存业务节点信息
        boolean result = exhibitionBoardService.save(exhibitionBoard);
        // 返回操作结果
        if (result) {
            for (String audioId : audioIdList) {
                final Asset asset = this.assetService.getById(audioId);
                asset.setFileType(FileTypeEnum.AUDIO.name());
                asset.setFileCat(FileCatEnum.EXHIBITION_BOARD_AUDIO.name());
                asset.setRefItemId(exhibitionBoard.getId());
                this.assetService.updateById(asset);
            }
            for (String datumId : datumIdList) {
                final Asset asset = this.assetService.getById(datumId);
                asset.setFileCat(FileCatEnum.EXHIBITION_BOARD_DATUM.name());
                asset.setRefItemId(exhibitionBoard.getId());
                this.assetService.updateById(asset);
            }

            final Audit audit = Audit.builder()
                    .content(exhibitionBoard.getName())
                    .refItemId(exhibitionBoard.getId())
                    .type(AuditTypeEnum.EXHIBITION_BOARD.name())
                    .operation(AuditOperationEnum.ADD.name())
                    .status(AuditStatusEnum.TBC.name())
                    .deleted(false)
                    .level(AuditStatusEnum.TBC.name())
                    .build();
            this.auditService.save(audit);
            return getSuccessResult();
        } else {
            // 保存失败
            return getFailResult();
        }
    }

    @PutMapping("/update")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:update")
    @ApiOperation(value = "修改展板信息", notes = "修改展板信息")
    public Map<String, Object> updateExhibitionBoard(@Validated(value = {Update.class}) ExhibitionBoard exhibitionBoard) {
        final VideoContent videoContent = this.videoContentService.getById(exhibitionBoard.getVideoContentId());
        exhibitionBoard.setVideoContentName(videoContent.getName());
        exhibitionBoard.setAuditStatus(AuditStatusEnum.TBC.name());
        exhibitionBoard.setPublished(false);
        boolean flag = exhibitionBoardService.updateById(exhibitionBoard);
        if (flag) {
            final List<String> audioIdList = exhibitionBoard.getAudioIdList();
            if (audioIdList != null && !audioIdList.isEmpty()) {
                final LambdaQueryWrapper<Asset> assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, exhibitionBoard.getId());
                assetQueryWrapper.eq(Asset::getFileCat, FileCatEnum.EXHIBITION_BOARD_AUDIO.name());
                final List<Asset> assetList = this.assetService.list(assetQueryWrapper);
                final Map<String, String> collect = assetList.stream().collect(Collectors.toMap(Asset::getId, Asset::getFileUrl));
                for (String audioId : audioIdList) {
                    final Asset asset = this.assetService.getById(audioId);
                    asset.setFileType(FileTypeEnum.AUDIO.name());
                    asset.setFileCat(FileCatEnum.EXHIBITION_BOARD_AUDIO.name());
                    asset.setRefItemId(exhibitionBoard.getId());
                    this.assetService.updateById(asset);
                    collect.remove(audioId);
                }
                collect.forEach((k, v) -> {
                    final boolean deleted = this.assetService.removeById(k);
                    if (deleted) {
                        FastDFSUtils.deleteFile(v);
                    }
                });
            }

            final List<String> datumIdList = exhibitionBoard.getDatumIdList();
            if (datumIdList != null && !datumIdList.isEmpty()) {
                final LambdaQueryWrapper<Asset> assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, exhibitionBoard.getId());
                assetQueryWrapper.eq(Asset::getFileCat, FileCatEnum.EXHIBITION_BOARD_DATUM.name());
                final List<Asset> assetList = this.assetService.list(assetQueryWrapper);
                final Map<String, String> collect = assetList.stream().collect(Collectors.toMap(Asset::getId, Asset::getFileUrl));
                for (String datumId : datumIdList) {
                    final Asset asset = this.assetService.getById(datumId);
                    asset.setFileCat(FileCatEnum.EXHIBITION_BOARD_DATUM.name());
                    asset.setRefItemId(exhibitionBoard.getId());
                    this.assetService.updateById(asset);
                    collect.remove(datumId);
                }
                collect.forEach((k, v) -> {
                    final boolean deleted = this.assetService.removeById(k);
                    if (deleted) {
                        FastDFSUtils.deleteFile(v);
                    }
                });
            }

            final Audit audit = Audit.builder()
                    .content(exhibitionBoard.getName())
                    .refItemId(exhibitionBoard.getId())
                    .type(AuditTypeEnum.EXHIBITION_BOARD.name())
                    .operation(AuditOperationEnum.EDIT.name())
                    .status(AuditStatusEnum.TBC.name())
                    .deleted(false)
                    .level(AuditStatusEnum.TBC.name())
                    .build();
            this.auditService.save(audit);
            return getSuccessResult();
        }
        return getFailResult();
    }

    @DeleteMapping("/delete/{id}")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:delete")
    @ApiOperation(value = "根据ID删除展板", notes = "根据ID删除展板")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "id", value = "标识ID", paramType = "path", dataType = "String")
    })
    public Map<String, Object> deleteExhibitionBoard(@PathVariable("id") String id) {
        final Audit audit = Audit.builder()
                .content(this.exhibitionBoardService.getById(id).getName())
                .refItemId(id)
                .type(AuditTypeEnum.EXHIBITION_BOARD.name())
                .operation(AuditOperationEnum.REMOVE.name())
                .status(AuditStatusEnum.TBC.name())
                .deleted(false)
                .level(AuditStatusEnum.TBC.name())
                .build();
        final boolean result = this.auditService.save(audit);
        if (result) {
            return getSuccessResult();
        }
        return getFailResult();
    }

    @GetMapping("/getList")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:list")
    @ApiOperation(value = "获取展板全部列表(无分页)", notes = "获取展板全部列表(无分页)")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "auditStatus", value = "审核状态", paramType = "query", dataType = "String")
    })
    public Map<String, Object> getExhibitionBoardList(
            @RequestParam(value = "exhibitionBoardCatIdList", required = false) List<String> exhibitionBoardCatIdList,
            @RequestParam(value = "boardCopyrightOwnerIdList", required = false) List<String> boardCopyrightOwnerIdList,
            @RequestParam(value = "auditStatus", defaultValue = "APPROVED_FINAL", required = false) AuditStatusEnum auditStatus) {
        final LambdaQueryWrapper<ExhibitionBoard> lambdaQueryWrapper = Wrappers.<ExhibitionBoard>lambdaQuery().eq(ExhibitionBoard::getAuditStatus, auditStatus.name()).eq(ExhibitionBoard::getPublished, true);
        if (exhibitionBoardCatIdList != null && !exhibitionBoardCatIdList.isEmpty()) {
            lambdaQueryWrapper.in(ExhibitionBoard::getExhibitionBoardCatId, exhibitionBoardCatIdList);
        }
        if (boardCopyrightOwnerIdList != null && !boardCopyrightOwnerIdList.isEmpty()) {
            lambdaQueryWrapper.in(ExhibitionBoard::getBoardCopyrightOwnerId, boardCopyrightOwnerIdList);
        }
        lambdaQueryWrapper.orderByDesc(ExhibitionBoard::getCreateTime);
        List<ExhibitionBoard> exhibitionBoardList = exhibitionBoardService.list();
        for (ExhibitionBoard exhibitionBoard : exhibitionBoardList) {
            if (exhibitionBoard.getBoardCopyrightOwnerId() != null) {
                String name = this.copyrightOwnerService.getById(exhibitionBoard.getBoardCopyrightOwnerId()).getName();
                exhibitionBoard.setBoardCopyrightOwnerName(name);
            }
            if (exhibitionBoard.getVideoContentCopyrightOwnerId() != null) {
                String name = this.copyrightOwnerService.getById(exhibitionBoard.getVideoContentCopyrightOwnerId()).getName();
                exhibitionBoard.setVideoContentCopyrightOwnerName(name);
            }
            if (exhibitionBoard.getExhibitionBoardCatId() != null) {
                String name = this.exhibitionBoardCatService.getById(exhibitionBoard.getExhibitionBoardCatId()).getName();
                exhibitionBoard.setExhibitionBoardCatName(name);
            }
        }
        return getResult(exhibitionBoardList);
    }

    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "_index", value = "分页起始偏移量", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "_size", value = "返回条数", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "nameOrCode", value = "名称或编码", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "copyrightOwner", value = "版权方", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "startDate", value = "创建时间-开始", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "endDate", value = "创建时间-结束", paramType = "query", dataType = "String")
    })
    @PostMapping("/getPageList")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:page")
    @ApiOperation(value = "获取展板分页列表", notes = "获取展板分页列表")
    public Map<String, Object> getExhibitionBoardPageList(GenericPageParam genericPageParam) {
        LambdaQueryWrapper<ExhibitionBoard> queryWrapper = new LambdaQueryWrapper<>();
        // 对名称或编码模糊查询
        if (StringUtils.isNotBlank(genericPageParam.getNameOrCode())) {
            queryWrapper.like(ExhibitionBoard::getName, genericPageParam.getNameOrCode());
        }
        // 对版权方模糊查询
        if (StringUtils.isNotBlank(genericPageParam.getBoardCopyrightOwnerId())) {
            queryWrapper.eq(ExhibitionBoard::getBoardCopyrightOwnerId, genericPageParam.getBoardCopyrightOwnerId());
        }
        // 根据创建时间区间检索
        if (genericPageParam.getStartDate() != null && genericPageParam.getEndDate() != null) {
            queryWrapper.ge(ExhibitionBoard::getCreateTime, genericPageParam.getStartDate().atTime(0, 0, 0))
                    .le(ExhibitionBoard::getCreateTime, genericPageParam.getEndDate().atTime(23, 59, 59));
        }
        // 设置排序规则
        queryWrapper.orderByDesc(ExhibitionBoard::getCreateTime);
        // 设置查询内容
        queryWrapper.select(
                ExhibitionBoard::getId,
                ExhibitionBoard::getName,
                ExhibitionBoard::getAuditStatus,
                ExhibitionBoard::getPublished,
                ExhibitionBoard::getDeleted,
                ExhibitionBoard::getVideoContentCopyrightOwnerId,
                ExhibitionBoard::getExhibitionBoardCatId,
                ExhibitionBoard::getCreateTime,
                ExhibitionBoard::getUpdateTime);
        Page<ExhibitionBoard> page = this.exhibitionBoardService.page(getPage(), queryWrapper);
        for (ExhibitionBoard exhibitionBoard : page.getRecords()) {
            if (exhibitionBoard.getBoardCopyrightOwnerId() != null) {
                String name = this.copyrightOwnerService.getById(exhibitionBoard.getBoardCopyrightOwnerId()).getName();
                exhibitionBoard.setBoardCopyrightOwnerName(name);
            }
            if (exhibitionBoard.getVideoContentCopyrightOwnerId() != null) {
                String name = this.copyrightOwnerService.getById(exhibitionBoard.getVideoContentCopyrightOwnerId()).getName();
                exhibitionBoard.setVideoContentCopyrightOwnerName(name);
            }
            if (exhibitionBoard.getExhibitionBoardCatId() != null) {
                String name = this.exhibitionBoardCatService.getById(exhibitionBoard.getExhibitionBoardCatId()).getName();
                exhibitionBoard.setExhibitionBoardCatName(name);
            }
            if (exhibitionBoard.getVideoContentId() != null) {
                String name = this.videoContentService.getById(exhibitionBoard.getVideoContentId()).getName();
                exhibitionBoard.setVideoContentName(name);
            }
        }
        return getResult(page);
    }

    @ApiOperation(value = "获取展板详情", notes = "获取展板详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "标识ID", dataType = "String", paramType = "path")
    })
    @GetMapping("/get/{id}")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:get:id")
    public Map<String, Object> getById(@PathVariable("id") String id) {
        ExhibitionBoard exhibitionBoard = exhibitionBoardService.getById(id);
        String exhibitionBoardCatId = exhibitionBoard.getExhibitionBoardCatId();
        if (exhibitionBoardCatId != null) {
            exhibitionBoard.setExhibitionBoardCatName(this.exhibitionBoardCatService.getById(exhibitionBoardCatId).getName());
        }
        String boardCopyrightOwnerId = exhibitionBoard.getBoardCopyrightOwnerId();
        if (boardCopyrightOwnerId != null) {
            exhibitionBoard.setBoardCopyrightOwnerName(this.copyrightOwnerService.getById(boardCopyrightOwnerId).getName());
        }
        if (exhibitionBoard.getVideoContentCopyrightOwnerId() != null) {
            String name = this.copyrightOwnerService.getById(exhibitionBoard.getVideoContentCopyrightOwnerId()).getName();
            exhibitionBoard.setVideoContentCopyrightOwnerName(name);
        }

        LambdaQueryWrapper<Asset> assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, exhibitionBoard.getId());
        assetQueryWrapper.eq(Asset::getFileCat, FileCatEnum.EXHIBITION_BOARD_AUDIO.name());
        final List<Asset> audioList = this.assetService.list(assetQueryWrapper);
        exhibitionBoard.setAudioList(audioList);

        assetQueryWrapper.clear();
        assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, exhibitionBoard.getId());
        assetQueryWrapper.eq(Asset::getFileCat, FileCatEnum.EXHIBITION_BOARD_DATUM.name());
        final List<Asset> datumList = this.assetService.list(assetQueryWrapper);
        exhibitionBoard.setDatumList(datumList);

        final LambdaQueryWrapper<Audit> auditQueryWrapper = Wrappers.<Audit>lambdaQuery().eq(Audit::getRefItemId, id);
        auditQueryWrapper.select(Audit::getContent);
        auditQueryWrapper.select(Audit::getType);
        auditQueryWrapper.select(Audit::getOperation);
        auditQueryWrapper.select(Audit::getStatus);
        auditQueryWrapper.select(Audit::getFirstTime);
        auditQueryWrapper.select(Audit::getFirstRemarks);
        auditQueryWrapper.select(Audit::getSecondTime);
        auditQueryWrapper.select(Audit::getSecondTime);
        auditQueryWrapper.select(Audit::getLevel);
        final List<Audit> auditList = this.auditService.list(auditQueryWrapper);
        exhibitionBoard.setAuditHistoryList(auditList);

        final String videoContentId = exhibitionBoard.getVideoContentId();
        if (videoContentId != null) {
            final VideoContent videoContent = this.videoContentService.getById(videoContentId);
            exhibitionBoard.setVideoContentName(videoContent.getName());

            assetQueryWrapper.clear();
            assetQueryWrapper = Wrappers.<Asset>lambdaQuery().eq(Asset::getRefItemId, videoContentId);
            assetQueryWrapper.eq(Asset::getFileCat, FileCatEnum.VIDEO_CONTENT.name());
            final List<Asset> videoList = this.assetService.list(assetQueryWrapper);
            exhibitionBoard.setVideoList(videoList);
        }
        return getResult(exhibitionBoard);
    }

    @ApiOperation(value = "上架/下架展板内容", notes = "上架/下架展板内容")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "标识ID", dataType = "String", paramType = "path"),
            @ApiImplicitParam(name = "isPublish", value = "是否上架", dataType = "boolean", paramType = "query", allowableValues = "True, False")
    })
    @PutMapping("/publish/{id}")
    @RequiresAuthentication  //@RequiresPermissions("exhibition:board:publish")
    public Map<String, Object> enableExhibitionBoard(@PathVariable("id") String id, @RequestParam("isPublish") Boolean isPublish) {
        final Audit audit = Audit.builder()
                .content(this.exhibitionBoardService.getById(id).getName())
                .refItemId(id)
                .type(AuditTypeEnum.EXHIBITION_BOARD.name())
                .operation(isPublish ? AuditOperationEnum.ENABLE.name() : AuditOperationEnum.DISABLE.name())
                .status(AuditStatusEnum.TBC.name())
                .deleted(false)
                .level(AuditStatusEnum.TBC.name())
                .build();
        this.auditService.save(audit);
        return getSuccessResult();
    }

}

