package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.core.annotations.OperationLog;
import cn.wisenergy.chnmuseum.party.model.Menu;
import cn.wisenergy.chnmuseum.party.model.RoleMenu;
import cn.wisenergy.chnmuseum.party.service.impl.MenuServiceImpl;
import cn.wisenergy.chnmuseum.party.service.impl.RoleMenuServiceImpl;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.*;

/**
 * <p>
 * 菜单权限 前端控制器
 * </p>
 *
 * @author 杨智平
 * @since 2018-08-03
 */
@Api(tags = "菜单权限接口")
@RestController
@RequestMapping("/menu")
public class MenuController extends BaseController {

    private static final Logger logger = LoggerFactory.getLogger(MenuController.class);

    @Resource
    private MenuServiceImpl menuService;

    @Resource
    private RoleMenuServiceImpl roleMenuService;

    /**
     * 获取单个菜单信息
     *
     * @param Id
     * @return
     */
    @ApiOperation(value = "获取单个菜单信息")
    @RequestMapping(value = "/get", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/menu/get")
    public ResponseEntity<Menu> get(String Id) {
        try {
            Menu menu = menuService.getById(Id);
            if (null == menu) {
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
            }
            return ResponseEntity.ok(menu);
        } catch (Exception e) {
            logger.error("查询菜单错误！", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /**
     * 查询菜单列表
     *
     * @param name
     * @return
     */
    @ApiOperation(value = "查询菜单列表")
    @RequestMapping(value = "/getMenuList", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/menu/getMenuList")
    public ResponseEntity<Page<Menu>> queryUserList(String name, String url) {
        try {
            QueryWrapper<Menu> ew = new QueryWrapper<>();
            if (StringUtils.isNotBlank(name)) {
                ew.like("menu_name", name);
            }
            if (StringUtils.isNotBlank(url)) {
                ew.like("menu_url", url);
            }
            ew.eq("is_deleted", 0).orderByAsc("sort");
            Page<Menu> page = getPage();
            Page<Menu> employeePage = menuService.page(page, ew);
            return ResponseEntity.ok(employeePage);
        } catch (Exception e) {
            logger.error("查询成员列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    // 添加
    @OperationLog("添加菜单")
    @ApiOperation(value = "添加")
    @PostMapping(value = "/add")
    @RequiresAuthentication  //@RequiresPermissions("/menu/add")
    public ResponseEntity<Map<String, Object>> add(Menu menu) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean ret = false;
            if (StringUtils.isBlank(menu.getName())) {
                resultMap.put("status", "400");
                resultMap.put("message", "菜单名称不能为空！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (StringUtils.isBlank(String.valueOf(menu.getUrl()))) {
                resultMap.put("status", "400");
                resultMap.put("message", "Url不能为空");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }

            if (!menu.getParentId().equals("0")) {
                Menu parent = this.menuService.getById(menu.getParentId());
                String parentName = parent.getName().trim();
                menu.setName(parentName + "-" + menu.getName().trim());
            }

            QueryWrapper<Menu> ew = new QueryWrapper<>();
            if (StringUtils.isNoneBlank(menu.getName())) {
                menu.setName(menu.getName().trim());
                ew.eq("menu_name", menu.getName());
                Menu one = this.menuService.getOne(ew);
                if (one != null) {
                    resultMap.put("status", "400");
                    resultMap.put("message", "菜单名称已存在！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                            resultMap);
                }
            }
            if (StringUtils.isNoneBlank(menu.getUrl())) {
                ew = new QueryWrapper<>();
                ew.eq("menu_url", menu.getUrl());
                Menu one = this.menuService.getOne(ew);
                if (one != null) {
                    resultMap.put("status", "400");
                    resultMap.put("message", "Url已存在！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                            resultMap);
                }
            }
            if (menu.getSort() != null) {
                ew = new QueryWrapper<>();
                ew.eq("sort", menu.getSort());
                ew.eq("is_deleted", 0);
                Menu one = this.menuService.getOne(ew);
                if (one != null) {
                    resultMap.put("status", "400");
                    resultMap.put("message", "顺序号已存在！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                            resultMap);
                }
            }

            menu.setCreateTime(new Date(System.currentTimeMillis()));
            menu.setUpdateTime(menu.getCreateTime());
            ret = this.menuService.save(menu);
            if (!ret) {
                // 更新失败, 400
                resultMap.put("status", "400");
                resultMap.put("message", "添加失败！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            // 204
            resultMap.put("status", "200");
            resultMap.put("message", "添加成功！");
            return ResponseEntity.ok(resultMap);
        } catch (Exception e) {
            logger.error("添加权限错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 修改
    @OperationLog("编辑菜单")
    @ApiOperation(value = "修改")
    @RequestMapping(value = "/edit", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/menu/edit")
    public ResponseEntity<Map<String, Object>> edit(Menu menu) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean ret = false;
            if (menu.getId() != null) {
                if (!menu.getParentId().equals("0")) {
                    Menu parent = this.menuService.getById(menu.getParentId());
                    String parentName = parent.getName();
                    menu.setName(parentName + "-" + menu.getName().trim());
                }
                QueryWrapper<Menu> ew = new QueryWrapper<>();
                if (StringUtils.isNoneBlank(menu.getName())) {
                    menu.setName(menu.getName().trim());
                    ew.ne("id", menu.getId());
                    ew.eq("menu_name", menu.getName());
                    Menu one = this.menuService.getOne(ew);
                    if (one != null) {
                        resultMap.put("status", "400");
                        resultMap.put("message", "菜单名称已存在！");
                        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                                resultMap);
                    }
                }
                if (StringUtils.isNoneBlank(menu.getUrl())) {
                    ew = new QueryWrapper<>();
                    ew.ne("id", menu.getId());
                    ew.eq("menu_url", menu.getUrl());
                    Menu one = this.menuService.getOne(ew);
                    if (one != null) {
                        resultMap.put("status", "400");
                        resultMap.put("message", "Url已存在！");
                        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                                resultMap);
                    }
                }
                if (menu.getSort() != null) {
                    ew = new QueryWrapper<>();
                    ew.ne("id", menu.getId());
                    ew.eq("sort", menu.getSort());
                    ew.eq("is_deleted", 0);
                    Menu one = this.menuService.getOne(ew);
                    if (one != null) {
                        resultMap.put("status", "400");
                        resultMap.put("message", "顺序号已存在！");
                        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                                resultMap);
                    }
                }
                ret = this.menuService.updateById(menu);
            } else {
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
            }
            if (!ret) {
                // 更新失败, 500
                resultMap.put("status", "500");
                resultMap.put("message", "更新错误！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", "200");
            resultMap.put("message", "更新成功！");
            return ResponseEntity.ok(resultMap);
        } catch (Exception e) {
            logger.error("编辑角色错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 刪除
    @OperationLog("删除菜单")
    @ApiOperation(value = "刪除")
    @RequestMapping(value = "/delete", method = RequestMethod.DELETE)
    @RequiresAuthentication  //@RequiresPermissions("/menu/delete")
    public ResponseEntity<Map<String, Object>> delete(@RequestParam(value = "menuId", required = true) String menuId) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean ret = false;
            if (StringUtils.isNotBlank(menuId)) {
                RoleMenu roleMenu = new RoleMenu();
                roleMenu.setMenuId(menuId);
                QueryWrapper<RoleMenu> ew = new QueryWrapper<RoleMenu>();
                ew.setEntity(roleMenu);
                List<RoleMenu> roleRightList = this.roleMenuService.list(ew);
                // 如果存在权限，先进行删除
                if (roleRightList.size() > 0) {
                    for (RoleMenu rm : roleRightList) {
                        this.roleMenuService.remove(new QueryWrapper<RoleMenu>(rm));
                    }
                }
                ret = this.menuService.removeById(menuId);
            } else {
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
            }
            if (!ret) {
                // 删除失败, 500
                resultMap.put("status", "500");
                resultMap.put("message", "删除失败！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", "200");
            resultMap.put("message", "删除成功！");
            return ResponseEntity.ok(resultMap);
        } catch (Exception e) {
            logger.error("删除菜单错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 批量刪除
    @OperationLog("批量删除菜单")
    @ApiOperation(value = "批量刪除")
    @RequestMapping(value = "/batchDel", method = RequestMethod.DELETE)
    @RequiresAuthentication  //@RequiresPermissions("/menu/batchDel")
    public ResponseEntity<Map<String, Object>> batchDel(@RequestParam(value = "menuIds", required = true) String menuIds) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean ret = false;
            if (StringUtils.isNotBlank(menuIds)) {
                QueryWrapper<RoleMenu> ew = new QueryWrapper<RoleMenu>();
                ew.in("menu_id", StringUtils.split(menuIds, ","));
                List<RoleMenu> roleRightList = this.roleMenuService.list(ew);
                // 如果存在权限，先进行删除
                if (roleRightList.size() > 0) {
                    for (RoleMenu rm : roleRightList) {
                        this.roleMenuService.remove(new QueryWrapper<RoleMenu>(rm));
                    }
                }

                ret = this.menuService.removeByIds(Arrays.asList(StringUtils.split(menuIds, ",")));
            } else {
                resultMap.put("status", "400");
                resultMap.put("message", "错误请求！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (!ret) {
                // 删除失败, 500
                resultMap.put("status", "500");
                resultMap.put("message", "删除失败！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", "200");
            resultMap.put("message", "删除成功！");
            return ResponseEntity.ok(resultMap);
        } catch (Exception e) {
            logger.error("删除角色错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    @ApiOperation(value = "获取角色对应菜单列表--弹框树使用")
    @RequestMapping(value = "/getRoleMenu", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/menu/getRoleMenu")
    public ResponseEntity<List<Map<String, String>>> getRoleMenu(String roleId) {
        try {
            QueryWrapper<Menu> menuWrapper = new QueryWrapper<>();
            menuWrapper.orderByAsc("sort");
            List<Menu> list = this.menuService.list(menuWrapper);
            List<Object> roleMenuList = null;
            if (roleId != null) {
                QueryWrapper<RoleMenu> wrapper = new QueryWrapper<>();
                wrapper.eq("role_id", roleId).select("menu_id");
                roleMenuList = this.roleMenuService.listObjs(wrapper);
            }

            List<Map<String, String>> menuList = new ArrayList<>();
            for (Menu m : list) {
                Map<String, String> map = new HashMap<>();
                map.put("id", m.getId());
                map.put("parentId", m.getParentId());
                map.put("menuName", m.getName());
                map.put("menuUrl", m.getUrl());
                map.put("sort", String.valueOf(m.getSort()));
                // 默认展开树
                map.put("open", "true");
                if (roleId != null) {
                    // 如果角色已有该菜单，则默认选中
                    if (roleMenuList.contains(m.getId())) {
                        map.put("checked", "true");
                    } else {
                        map.put("checked", "false");
                    }
                }
                menuList.add(map);
            }
            return ResponseEntity.ok(menuList);
        } catch (Exception e) {
            logger.error("获取该角色菜单失败", e.getMessage());
            e.printStackTrace();
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    /*@ApiOperation(value = "登录后获取当前用户的角色对应权限菜单列表")
    @RequestMapping(value = "/getUserMenuPerms", method = RequestMethod.GET)
    public ResponseEntity<List<Menu>> getUserMenuPerms(String roleId) {
        try {
            if (roleId != null) {
                List<Menu> menuList = this.menuService.getUserMenuPerms(roleId);
                return ResponseEntity.ok(menuList);
            }
        } catch (Exception e) {
            logger.error("获取该角色菜单失败", e.getMessage());
            e.printStackTrace();
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }*/
}

