package cn.wisenergy.chnmuseum.party.web.controller;

import cn.wisenergy.chnmuseum.party.core.annotations.OperationLog;
import cn.wisenergy.chnmuseum.party.model.Permission;
import cn.wisenergy.chnmuseum.party.model.RolePermission;
import cn.wisenergy.chnmuseum.party.service.PermissionService;
import cn.wisenergy.chnmuseum.party.service.RolePermissionService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import cn.wisenergy.chnmuseum.party.web.controller.base.BaseController;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;

/**
 * 角色管理Controller
 */
@RestController
@RequestMapping(value = "/permission")
public class PermissionController extends BaseController {

    private static final Logger logger = LoggerFactory.getLogger(PermissionController.class);

    @Autowired
    private PermissionService permissionService;

    @Autowired
    private RolePermissionService rolePermissionService;

    /**
     * 获取单个权限
     */
    @ApiOperation(value = "获取单个权限")
    @RequestMapping(value = "/getById", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/permission/getById")
    public ResponseEntity<Permission> getById(String Id) {
        try {
            Permission one = permissionService.getById(Id);
            if (null == one) {
                return ResponseEntity.status(HttpStatus.NOT_FOUND).body(null);
            }
            return ResponseEntity.ok(one);
        } catch (Exception e) {
            logger.error("查询错误！", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    // 获取的顶级权限列表
    @ApiOperation(value = "获取的顶级权限列表")
    @RequestMapping(value = "/getTopPermissionList", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/permission/getPermissionList")
    public ResponseEntity<List<Permission>> getTopPermissionList() {
        try {
            QueryWrapper<Permission> wrapper = new QueryWrapper<>();
            wrapper.eq("pid", 0).orderByAsc("create_time");
            List<Permission> permissionList = this.permissionService.list(wrapper);
            return ResponseEntity.ok(permissionList);
        } catch (Exception e) {
            logger.error("查询角色列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }


    // 获取权限分页对象
    @ApiOperation(value = "获取权限分页对象")
    @RequestMapping(value = "/getPermissionList", method = RequestMethod.GET)
    @RequiresAuthentication  //@RequiresPermissions("/permission/getPermissionList")
    public ResponseEntity<Page<Permission>> getPermissionList(QueryWrapper<Permission> ew, String name, String math) {
        try {
            Page<Permission> page = this.getPage();
            ew.orderByDesc("create_time");
            if (StringUtils.isNotBlank(name)) {
                ew.like("name", name.trim());
            }
            if (StringUtils.isNotBlank(math) && StringUtils.isNotEmpty(math)) {
                ew.like("url", math.trim());
            }

            Page<Permission> list = this.permissionService.page(page, ew);
            return ResponseEntity.ok(list);
        } catch (Exception e) {
            logger.error("查询角色列表出错!", e);
        }
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(null);
    }

    @OperationLog("添加权限")
    @ApiOperation(value = "添加")
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    @RequiresAuthentication  //@RequiresPermissions("/permission/add")
    public ResponseEntity<Map<String, Object>> add(Permission permission) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            boolean ret = false;
            if (!permission.getPid().equals(0)) {
                Permission parent = this.permissionService.getById(permission.getPid());
                String parentName = parent.getName().trim();
                permission.setName(parentName + "-" + permission.getName().trim());
            }
            QueryWrapper<Permission> ew = new QueryWrapper<>();
            if (StringUtils.isNoneBlank(permission.getName())) {
                permission.setName(permission.getName().trim());
                ew.eq("name", permission.getName());
                Permission one = this.permissionService.getOne(ew);
                if (one != null) {
                    resultMap.put("status", 400);
                    resultMap.put("message", "该权限已存在！");
                    return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                }
            }
            permission.setUrl(permission.getUrl().trim());
            permission.setCreateTime(new Date(System.currentTimeMillis()));
            permission.setUpdateTime(permission.getCreateTime());
            ret = this.permissionService.save(permission);
            if (!ret) {
                // 更新失败, 400
                resultMap.put("status", 400);
                resultMap.put("message", "添加失败！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "添加成功！");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("添加权限错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 修改
    @OperationLog("修改权限")
    @ApiOperation(value = "修改")
    @RequestMapping(value = "/edit", method = RequestMethod.PUT)
    @RequiresAuthentication  //@RequiresPermissions("/permission/edit")
    public ResponseEntity<Map<String, Object>> edit(Permission permission) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            boolean ret = false;
            if (permission.getId() != null) {
                if (!permission.getPid().equals(0)) {
                    Permission parent = this.permissionService.getById(permission.getPid());
                    String parentName = parent.getName().trim();
                    permission.setName(parentName + "-" + permission.getName().trim());
                }
                QueryWrapper<Permission> ew = new QueryWrapper<>();
                if (StringUtils.isNoneBlank(permission.getName())) {
                    permission.setName(permission.getName().trim());
                    ew.ne("id",permission.getId());
                    ew.eq("name", permission.getName());
                    Permission one = this.permissionService.getOne(ew);
                    if (one != null) {
                        resultMap.put("status", 400);
                        resultMap.put("message", "该权限已存在！");
                        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
                    }
                }
                permission.setUrl(permission.getUrl().trim());
                ret = this.permissionService.updateById(permission);
            } else {
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
            }
            if (!ret) {
                // 更新失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "更新错误！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "更新成功！");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("编辑角色错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 刪除
    @OperationLog("删除权限")
    @ApiOperation(value = "刪除")
    @RequestMapping(value = "/delete", method = RequestMethod.DELETE)
    @RequiresAuthentication  //@RequiresPermissions("/permission/delete")
    public ResponseEntity<Map<String, Object>> delete(@RequestParam(value = "permissionId", required = true) String permissionId) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            boolean ret = false;
            if (StringUtils.isNotBlank(permissionId)) {
                RolePermission rolePermission = new RolePermission();
                rolePermission.setPid(permissionId);
                QueryWrapper<RolePermission> ew = new QueryWrapper<RolePermission>();
                ew.setEntity(rolePermission);
                List<RolePermission> roleRightList = this.rolePermissionService.list(ew);
                // 如果存在权限，先进行删除
                if (roleRightList.size() > 0) {
                    for (RolePermission rp : roleRightList) {
                        this.rolePermissionService.remove(new QueryWrapper<RolePermission>(rp));
                    }
                }
                ret = this.permissionService.removeById(permissionId);
            } else {
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).build();
            }
            if (!ret) {
                // 删除失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "删除失败！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "删除成功！");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("删除权限错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

    // 批量刪除
    @OperationLog("批量删除权限")
    @ApiOperation(value = "批量刪除")
    @RequestMapping(value = "/batchDel", method = RequestMethod.DELETE)
    @RequiresAuthentication  //@RequiresPermissions("/permission/batchDel")
    public ResponseEntity<Map<String, Object>> batchDel(@RequestParam(value = "permissionIds") String permissionIds) {
        Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
        try {
            boolean ret = false;
            if (StringUtils.isNotBlank(permissionIds)) {
                QueryWrapper<RolePermission> ew = new QueryWrapper<RolePermission>();
                ew.in("pid", StringUtils.split(permissionIds, ","));
                List<RolePermission> roleRightList = this.rolePermissionService.list(ew);
                // 如果存在权限，先进行删除
                if (roleRightList.size() > 0) {
                    for (RolePermission rp : roleRightList) {
                        this.rolePermissionService.remove(new QueryWrapper<>(rp));
                    }
                }

                ret = this.permissionService.removeByIds(Arrays.asList(StringUtils.split(permissionIds, ",")));
            } else {
                resultMap.put("status", 400);
                resultMap.put("message", "错误请求！");
                return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(resultMap);
            }
            if (!ret) {
                // 删除失败, 500
                resultMap.put("status", 500);
                resultMap.put("message", "删除失败！");
                return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(resultMap);
            }
            // 204
            resultMap.put("status", 200);
            resultMap.put("message", "删除成功！");
            return ResponseEntity.status(HttpStatus.OK).body(resultMap);
        } catch (Exception e) {
            logger.error("删除角色错误!", e);
        }
        // 500
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
    }

}
