package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.utils.DateUtil;
import cn.wisenergy.common.utils.R;
import cn.wisenergy.mapper.AccountMapper;
import cn.wisenergy.mapper.TradeRecordMapper;
import cn.wisenergy.model.app.AccountInfo;
import cn.wisenergy.model.app.TradeRecord;
import cn.wisenergy.model.enums.TradeRecordEnum;
import cn.wisenergy.model.vo.*;
import cn.wisenergy.service.app.WalletService;
import io.swagger.models.auth.In;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author 86187
 */
@Service
@Slf4j
public class WalletServiceImpl implements WalletService {
    private static final String PATTERN = "yyyy-MM";

    @Autowired
    private AccountMapper accountMapper;

    @Autowired
    private TradeRecordMapper tradeRecordMapper;


    @Override
    public R<MoneyPackageVo> getMoneyPackage(String userId) {
        log.info("shop-mall[]WalletServiceImpl[]getMoneyPackage[]input.param.userId:" + userId);
        if (StringUtils.isBlank(userId)) {
            return R.error("入参为空!");
        }

        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);

        //1、获取用户账户信息
        AccountInfo accountInfo = accountMapper.getByUserIdAndTime(userId, yearMonth);
        if (null == accountInfo) {
            return R.error("用户账户信息不存在!");
        }
        MoneyPackageVo moneyPackageVo = new MoneyPackageVo();
        moneyPackageVo.setUserId(accountInfo.getUserId());
        moneyPackageVo.setMoneyIncome(accountInfo.getEarningsMonth());
        moneyPackageVo.setTotalIncome(accountInfo.getEarningsTotal());
        moneyPackageVo.setCurrentMoneyCan(accountInfo.getExtractMoney());

        return R.ok(moneyPackageVo);
    }

    @Override
    public R<WithdrawalAmountVo> getWithdrawalAmount(String userId) {
        log.info("shop-mall[]WalletServiceImpl[]getWithdrawalAmount[]input.param.userId:" + userId);
        if (StringUtils.isBlank(userId)) {
            return R.error("入参为空!");
        }

        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);

        //1、获取用户账户信息
        AccountInfo accountInfo = accountMapper.getByUserIdAndTime(userId, yearMonth);
        if (null == accountInfo) {
            return R.error("用户账户信息不存在!");
        }

        WithdrawalAmountVo withdrawalAmountVo = new WithdrawalAmountVo();
        withdrawalAmountVo.setUserId(userId);
        withdrawalAmountVo.setCurrentMoneyCan(accountInfo.getExtractMoney());
        //TODO  获取上月为提现
        withdrawalAmountVo.setLastMoneyNot(new BigDecimal(0));
        return R.ok(withdrawalAmountVo);
    }

    @Override
    public R<List<AccumulatedIncomeVo>> showIncomeRecord(String userId) {
        log.info("shop-mall[]WalletServiceImpl[]showIncomeRecord[]input.param.userId:" + userId);
        if (StringUtils.isBlank(userId)) {
            return R.error("入参为空!");
        }

        List<TradeRecord> list = tradeRecordMapper.getByUserId(userId);
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(new ArrayList<>());
        }

        List<AccumulatedIncomeVo> result = new ArrayList<>();
        for (TradeRecord tradeRecord : list) {
            AccumulatedIncomeVo incomeVo = new AccumulatedIncomeVo();
            incomeVo.setUserId(tradeRecord.getUserId());
            String yearMonth = DateUtil.convertDateToStr(tradeRecord.getCreateTime(), PATTERN);
            incomeVo.setYearMonth(yearMonth);
            incomeVo.setIncome(tradeRecord.getMoney());
            result.add(incomeVo);
        }

        return R.ok(result);
    }

    @Override
    public R<MoneyPackageDetailVo> queryIncomeDetail(String userId) {
        log.info("shop-mall[]WalletServiceImpl[]queryIncomeDetail[]input.param.userId:" + userId);
        if (StringUtils.isBlank(userId)) {
            return R.error("入参为空!");
        }

        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);
        //获取当月账户信息
        AccountInfo accountInfo = accountMapper.getByUserIdAndTime(userId, yearMonth);
        if (null == accountInfo) {
            return R.error("账户信息为空!");
        }

        //获取本月交易记录
        List<TradeRecord> list = tradeRecordMapper.getByUserIdAndTime(userId, yearMonth);
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(new MoneyPackageDetailVo());
        }

        MoneyPackageDetailVo detailVo = new MoneyPackageDetailVo();
        detailVo.setUserId(userId);
        detailVo.setTotalIncome(accountInfo.getEarningsTotal());

        List<IncomeDetailVo> incomeDetailVos = new ArrayList<>();
        for (TradeRecord tradeRecord : list) {
            IncomeDetailVo incomeDetailVo = new IncomeDetailVo();
            String typeName = TradeRecordEnum.getByCode(tradeRecord.getTradeType());

            incomeDetailVo.setTypeName(typeName);
            incomeDetailVo.setMoney(tradeRecord.getMoney());
            incomeDetailVo.setIncomeTime(tradeRecord.getCreateTime());
            incomeDetailVos.add(incomeDetailVo);
        }

        detailVo.setList(incomeDetailVos);

        return R.ok(detailVo);
    }
}
