package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.utils.R;
import cn.wisenergy.mapper.AccountMapper;
import cn.wisenergy.mapper.MemberPercentMapper;
import cn.wisenergy.mapper.TeamPerformanceMapper;
import cn.wisenergy.mapper.UsersMapper;
import cn.wisenergy.model.app.*;
import cn.wisenergy.model.enums.MonthlyTaskStatus;
import cn.wisenergy.model.enums.RebateStatusEnum;
import cn.wisenergy.model.vo.TeamPerformanceSortVo;
import cn.wisenergy.service.Manager.PublicManager;
import cn.wisenergy.service.app.AccountService;
import cn.wisenergy.service.Manager.AccountManager;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;


/**
 * @author 86187
 */
@Slf4j
@Service
public class AccountServiceImpl extends ServiceImpl<AccountMapper, AccountInfo> implements AccountService {

    @Autowired
    private UsersMapper usersMapper;

    @Autowired
    private AccountMapper accountMapper;

    @Autowired
    private MemberPercentMapper memberPercentMapper;

    @Autowired
    private AccountManager accountManager;

    @Autowired
    private PublicManager publicManager;

    @Autowired
    private TeamPerformanceMapper teamPerformanceMapper;

    private static final String PATTERN = "yyyy-mm";

    @Override
    public R<Boolean> orderRebate(List<OrderInfo> list) {
        log.info("shop-mall[]AccountServiceImpl[]orderRebate[]input.param.list:{}", list.size());
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(0, true);
        }

        //1、遍历订单，订单状态是未返佣，完成时间小于当前时间 --返佣
        for (OrderInfo orderInfo : list) {
            long successTime = orderInfo.getSuccessTime().getTime();
            long time = System.currentTimeMillis();
            if (RebateStatusEnum.NO_REBATE.getCode().equals(orderInfo.getRebateStatus()) && successTime <= time) {
                //获取用户信息
                User user = usersMapper.selectById(orderInfo.getBuyerId());
                if (null == user) {
                    continue;
                }

                //获取账户信息
                AccountInfo accountInfo = accountMapper.getByUserId(user.getUserId());
                if (null == accountInfo) {
                    continue;
                }

                //获取返佣比例
                MemberPercent memberpercent = memberPercentMapper.getByLevel(user.getUserLevel());
                if (null == memberpercent) {
                    continue;
                }

                //修改订单状态为已返佣，账户可用金额增加
                accountManager.updateOrderAddMoney(orderInfo, accountInfo, memberpercent);
            }
        }
        return R.ok(0, true);
    }

    @Override
    public R<AccountInfo> getByUserId(String userId) {
        AccountInfo accountInfo = accountMapper.getByUserId(userId);
        return R.ok(accountInfo);
    }

    @Override
    public R<Boolean> performanceCount(List<OrderInfo> list) {
        log.info("shop-mall[]AccountServiceImpl[]performanceCount[]input.param.list:{}", list.size());
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(0, true);
        }

        //计算当月所有订单成交额
        BigDecimal totalMoney = new BigDecimal(0);
        for (OrderInfo orderInfo : list) {
            //判断是否是本月
            boolean bool = publicManager.isThisMonth(orderInfo.getCreateTime(), PATTERN);
            if (bool && orderInfo.getMonthlyTaskStatus() == 0) {
                totalMoney = totalMoney.add(orderInfo.getPayment());
            }
        }

        //遍历订单  订单状态创建时间，小于当前时间  订单当月处理状态 ：未处理
        for (OrderInfo orderInfo : list) {
            long createTime = orderInfo.getCreated().getTime();
            long time = System.currentTimeMillis();
            if (MonthlyTaskStatus.NO_DEAL_WITH.getCode().equals(orderInfo.getMonthlyTaskStatus()) && createTime <= time) {
                //获取用户信息
                User user = usersMapper.selectById(orderInfo.getBuyerId());
                if (null == user) {
                    continue;
                }

                List<TeamPerformance> teamPerformances = new ArrayList<>();

                //获取团队业绩信息
                TeamPerformance teamPerformance = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), new Date());
                if (null == teamPerformance) {
                    continue;
                }

                //1、统计当前用户月度业绩
                BigDecimal userCount = teamPerformance.getMonthTeamPerformance().add(orderInfo.getPayment());
                teamPerformance.setMonthTeamPerformance(userCount);
                teamPerformances.add(teamPerformance);

                //2、获取当前用户的上级用户列表
                List<User> userList = getByList(user.getUserId());
                if (CollectionUtils.isEmpty(userList)) {
                    continue;
                }

                for (User userInfo : userList) {
                    //3、统计当前用户上级月度绩效
                    TeamPerformance team = teamPerformanceMapper.getByUserIdAndTime(userInfo.getUserId(), new Date());
                    if (null == team) {
                        continue;
                    }
                    //1、统计当前用户月度绩效
                    BigDecimal monthCount = team.getMonthTeamPerformance().add(orderInfo.getPayment());
                    team.setMonthTeamPerformance(monthCount);
                    teamPerformances.add(team);
                }

                //4、更新账户月度绩效
                accountManager.updateAccountPerformanceMonth(teamPerformances);
            }

            //5、获取所有用户,如果会员等级是黄金以上，计算月度收益
            List<User> userList = usersMapper.getAllGoldUser();
            for (User user : userList) {
                monthlyIncome(totalMoney, user);
            }
        }
        return R.ok(0, true);

    }

    /**
     * 获取用户的商机信息
     *
     * @param userId 用户id
     * @return
     */
    @Override
    public List<User> getByList(String userId) {
        List<User> list = new ArrayList<>();

        getUser(list, userId);

        return list;
    }

    @Override
    public R<Boolean> progressPrizeCount() {
        log.info("shop-mall[]AccountServiceImpl[]performanceCount[]input.method");

        //1、判断当前月是否是业绩开始的第一个月
        List<TeamPerformance> teamPerformances = teamPerformanceMapper.getByBeforeTime(new Date());

        //获取当月所有人业绩总额
        Double totalMoney = teamPerformanceMapper.countByTime(new Date());

        //2、集合为空 是业绩开始的第一个月
        if (CollectionUtils.isEmpty(teamPerformances)) {
            double number = Math.floor(totalMoney / 3980 / 12);
            if (number != 0) {
                //获取月业绩前20用户
                List<TeamPerformance> list = teamPerformanceMapper.userTwenty(new Date());
                if (CollectionUtils.isEmpty(list)) {
                    return R.ok(0, true);
                }

                //获取20名进步最大的月业绩和
                Double twentyTotal = teamPerformanceMapper.countTwenty(new Date());
                for (TeamPerformance user : list) {
                    //获取最大进步奖 百分比
                    MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(user.getUserLevel(), 3);

                    //计算收益
                    double userTeamPerformance = user.getMonthTeamPerformance().doubleValue();
                    double percent = memberPercent.getPercent().doubleValue();

                    double income = number * 3980 * percent * userTeamPerformance / twentyTotal;

                    //获取账户信息
                    AccountInfo accountInfo = accountMapper.getByUserId(user.getUserId());
                    accountInfo.setEarningsMonth(new BigDecimal(income));

                    //更新用户账户表
                    accountMapper.updateById(accountInfo);

                }
            }
            return R.ok(0, true);
        }

        //3、集合不为空 不是业绩开始的第一个月
        //获取用户列表
        List<User> userList = usersMapper.findAll();
        if (CollectionUtils.isEmpty(userList)) {
            return R.ok(0, true);
        }

        //遍历 计算业绩增长率
        List<TeamPerformanceSortVo> listVo = new ArrayList<>();
        for (User user : userList) {
            TeamPerformanceSortVo teamVo = new TeamPerformanceSortVo();
            double growthRate;
            //获取当月业绩
            TeamPerformance teamPerformance = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), new Date());

            //获取上月业绩
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date());
            cal.add(Calendar.MONTH, -1);
            Date date = cal.getTime();
            TeamPerformance lastMonth = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), date);
            if (null == teamPerformance || null == lastMonth) {
                growthRate = 0.00;
            } else {
                double month = teamPerformance.getMonthTeamPerformance().doubleValue();
                double last = lastMonth.getMonthTeamPerformance().doubleValue();
                if (last >= month) {
                    growthRate = 0.00;
                } else {
                    growthRate = (month - last) / month;
                }
            }
            teamVo.setGrowthRate(growthRate);
            double monthPerformance;
            if (null == teamPerformance.getMonthTeamPerformance()) {
                monthPerformance = 0.00;
            } else {
                monthPerformance = teamPerformance.getMonthTeamPerformance().doubleValue();
            }
            teamVo.setMonthPerformance(monthPerformance);
            teamVo.setTeamPerformance(teamPerformance);
            listVo.add(teamVo);
        }

        //对集合进行排序
        listVo.sort(Comparator.comparing(TeamPerformanceSortVo::getGrowthRate).reversed());

        //计算前20的总业绩
        double total = listVo.stream().mapToDouble(TeamPerformanceSortVo::getMonthPerformance).sum();
        double number = Math.floor(totalMoney / 3980 / 12);
        if (listVo.size() >= 20) {
            //取排名前20的
            listVo.subList(0, 20);
            for (TeamPerformanceSortVo sortVo : listVo) {
                String userId = sortVo.getTeamPerformance().getUserId();
                Integer userLevel = sortVo.getTeamPerformance().getUserLevel();
                //获取最大进步奖 百分比
                MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(userLevel, 3);

                //计算收益
                double userTeamPerformance = sortVo.getTeamPerformance().getMonthTeamPerformance().doubleValue();
                double percent = memberPercent.getPercent().doubleValue();

                double income = number * 3980 * percent * userTeamPerformance / total;

                //获取账户信息
                AccountInfo accountInfo = accountMapper.getByUserId(userId);
                accountInfo.setEarningsMonth(new BigDecimal(income));

                //更新用户账户表
                accountMapper.updateById(accountInfo);
            }
            return R.ok(0, true);
        } else {
            //计算总业绩
            for (TeamPerformanceSortVo sortVo : listVo) {
                String userId = sortVo.getTeamPerformance().getUserId();
                Integer userLevel = sortVo.getTeamPerformance().getUserLevel();
                //获取最大进步奖 百分比
                MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(userLevel, 3);

                //计算收益
                double userTeamPerformance = sortVo.getTeamPerformance().getMonthTeamPerformance().doubleValue();
                double percent = memberPercent.getPercent().doubleValue();

                double income = number * 3980 * percent * userTeamPerformance / total;

                //获取账户信息
                AccountInfo accountInfo = accountMapper.getByUserId(userId);
                accountInfo.setEarningsMonth(new BigDecimal(income));

                //更新用户账户表
                accountMapper.updateById(accountInfo);
            }
        }
        return R.ok(0, true);
    }

    public void getUser(List<User> list, String userId) {
        User user = usersMapper.getByUserId(userId);
        list.add(user);
        if (null != user && StringUtils.isBlank(user.getBeInvitedCode())) {
            User userInfo = usersMapper.getByBeInvitedCode(user.getBeInvitedCode());
            getUser(list, userInfo.getUserId());
        }
    }

    /**
     * 如果会员等级是黄金以上，计算月度收益
     *
     * @param totalMoney
     * @param user
     * @return
     */
    private boolean monthlyIncome(BigDecimal totalMoney, User user) {
        double total = totalMoney.doubleValue();
        BigDecimal money;
        //当月所有人订单成交总金额/3980/12  向下整
        double moneyMonth = Math.floor(total / 3980 / 12);
        if (moneyMonth != 0) {
            //获取等级优惠百分比
            MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(user.getUserLevel(), 2);
            if (null != memberPercent) {
                //获取本月团队总金额
                TeamPerformance teamPerformance = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), new Date());

                //获取该等级团队总金额
                Double teamTotal = teamPerformanceMapper.countTeamMoney(user.getUserLevel(), new Date());
                if (null != teamPerformance && null != teamTotal) {
                    double month = teamPerformance.getMonthTeamPerformance().doubleValue();
                    double percent = memberPercent.getPercent().doubleValue();
                    //计算收益
                    double income = moneyMonth * 3980 * percent * month / teamTotal;

                    //获取账户信息
                    AccountInfo accountInfo = accountMapper.getByUserId(user.getUserId());
                    BigDecimal bigDecimal = new BigDecimal(income);
                    BigDecimal performanceMonth = accountInfo.getEarningsMonth().add(bigDecimal);
                    accountInfo.setEarningsMonth(performanceMonth);

                    BigDecimal performanceTotal = accountInfo.getEarningsMonth().add(bigDecimal);
                    accountInfo.setEarningsMonth(performanceTotal);

                    //更新月收益
                    int count = accountMapper.updateById(accountInfo);
                    if (count != 0) {
                        return true;
                    }
                }
            }

        }
        return false;
    }

}
