package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.utils.DateUtil;
import cn.wisenergy.common.utils.R;
import cn.wisenergy.mapper.*;
import cn.wisenergy.model.app.*;

import cn.wisenergy.model.enums.*;
import cn.wisenergy.model.vo.TeamPerformanceSortVo;
import cn.wisenergy.service.Manager.PublicManager;
import cn.wisenergy.service.app.AccountService;
import cn.wisenergy.service.Manager.AccountManager;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xxl.job.core.handler.annotation.XxlJob;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @author 86187
 */
@Slf4j
@Service
public class AccountServiceImpl extends ServiceImpl<AccountMapper, AccountInfo> implements AccountService {

    @Autowired
    private UsersMapper usersMapper;

    @Autowired
    private AccountMapper accountMapper;

    @Autowired
    private MemberPercentMapper memberPercentMapper;

    @Autowired
    private AccountManager accountManager;

    @Autowired
    private PublicManager publicManager;

    @Autowired
    private TradeRecordMapper tradeRecordMapper;

    @Autowired
    private TeamPerformanceMapper teamPerformanceMapper;

    @Autowired
    private ProgressPrizeMapper progressPrizeMapper;

    @Autowired
    private OrderMapper orderMapper;

    @Autowired
    private MonthManureMapper monthManureMapper;

    private static final String PATTERN = "yyyy-MM";

    private static final Integer TWENTY = 20;

    @XxlJob(value = "orderRebateDayTask")
    @Override
    public R<Boolean> orderRebate() {
        log.info("订单返佣接口定时-日任务");
        //        //获取返佣订单
        List<OrderInfo> list = orderMapper.getListBySuccessTime(new Date());
        log.info("shop-mall[]AccountServiceImpl[]orderRebate[]input.param.list:{}", list.size());
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(0, true);
        }

        //1、遍历订单，订单状态是未返佣，完成时间小于当前时间 --返佣
        for (OrderInfo orderInfo : list) {
            long successTime = orderInfo.getSuccessTime().getTime();
            long time = System.currentTimeMillis();
            if (RebateStatusEnum.NO_REBATE.getCode().equals(orderInfo.getRebateStatus()) && successTime <= time) {
                //获取用户信息
                User user = usersMapper.getByUserId(orderInfo.getUserId());
                if (null == user) {
                    continue;
                }

                //获取账户信息
                AccountInfo accountInfo = accountMapper.getByUserId(user.getUserId());
                if (null == accountInfo) {
                    continue;
                }

                //获取返佣比例
                MemberPercent memberpercent = memberPercentMapper.getByLevelAndType(user.getUserLevel(), MemberPercentEnum.REBATE.getCode());
                if (null == memberpercent) {
                    continue;
                }

                //修改订单状态为已返佣，账户可用金额增加
                accountManager.updateOrderAddMoney(orderInfo, accountInfo, memberpercent);
            }
        }
        return R.ok(0, true);
    }

    @Override
    public R<AccountInfo> getByUserId(String userId) {
        AccountInfo accountInfo = accountMapper.getByUserId(userId);
        return R.ok(accountInfo);
    }

    @XxlJob(value = "monthManureDayTask")
    @Override
    public R<Boolean> performanceCount() {
        //获取本月订单
        log.info("月度肥料定时-日任务");
        List<OrderInfo> list = orderMapper.getByCreateTime(new Date());
        log.info("shop-mall[]AccountServiceImpl[]performanceCount[]input.param.list:{}", list.size());
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(0, true);
        }

        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);

        //计算当月所有订单成交额
        BigDecimal totalMoney = new BigDecimal(0);

        //统计出出每个用户当月订单成交额 key:userId  value:用户当月订单成交额
        Map<String, Double> map = new HashMap<>();
        for (OrderInfo orderInfo : list) {
            String userId = orderInfo.getUserId();
            double payMoney = orderInfo.getPayment().doubleValue();

            //key 存在  累加订单金额 到 value
            if (map.containsKey(userId)) {
                double money = payMoney + map.get(orderInfo.getUserId());
                map.put(orderInfo.getUserId(), money);
            } else {
                //key 不存在，加入集合
                map.put(userId, payMoney);
            }

            //累加订单成交额
            totalMoney = totalMoney.add(orderInfo.getPayment());
        }

        //累计用户和上级用户-团队业绩
        Map<String, Double> tempMap = new HashMap<>();

        //遍历订单
        for (Map.Entry<String, Double> entity : map.entrySet()) {
            String userId = entity.getKey();
            //1)、统计当前用户月度业绩
            double userCount = entity.getValue();
            if (tempMap.containsKey(entity.getKey())) {
                double teamMoney = userCount + tempMap.get(entity.getKey());
                tempMap.put(entity.getKey(), teamMoney);
            } else {
                //key 不存在，加入集合 当前用户团队绩效
                tempMap.put(entity.getKey(), userCount);
            }

            //2)、获取当前用户的上级用户列表
            List<User> userList = getByList(userId);
            if (CollectionUtils.isEmpty(userList)) {
                continue;
            }

            for (User userInfo : userList) {
                //3)、统计当前用户的上级用户团队绩效
                //key 存在  当前用户团队绩效  + 上级用户团队绩效
                if (tempMap.containsKey(userInfo.getUserId())) {
                    double teamMoney = userCount + tempMap.get(userInfo.getUserId());
                    tempMap.put(userInfo.getUserId(), teamMoney);
                } else {
                    //key 不存在，加入集合 当前用户团队绩效
                    tempMap.put(userInfo.getUserId(), userCount);
                }
            }
        }

        //3、获取用户当月绩效信息   新增 or 更新
        List<TeamPerformance> addList = new ArrayList<>();
        List<TeamPerformance> updateList = new ArrayList<>();
        for (Map.Entry<String, Double> entity : tempMap.entrySet()) {
            //获取团队业绩信息
            TeamPerformance teamPerformance = teamPerformanceMapper.getByUserIdAndTime(entity.getKey(), yearMonth);
            if (null == teamPerformance) {
                //获取用户信息
                User user = usersMapper.getByUserId(entity.getKey());

                //添加用户团队业绩信息
                TeamPerformance performance = new TeamPerformance();
                performance.setUserId(user.getUserId());
                performance.setMonthTeamPerformance(BigDecimal.valueOf(entity.getValue()));
                performance.setUserLevel(user.getUserLevel());
                performance.setYearMonth(yearMonth);
                addList.add(performance);
            } else {
                teamPerformance.setMonthTeamPerformance(BigDecimal.valueOf(entity.getValue()));
                updateList.add(teamPerformance);
            }
        }

        //4、更新账户月度绩效
        boolean updateBool = accountManager.updateAccountPerformanceMonth(addList, updateList);
        if (!updateBool) {
            return R.ok(1, false);
        }

        //5、获取所有用户,如果会员等级是黄金以上，计算月度收益
        List<User> userList = usersMapper.getAllGoldUser();
        if (CollectionUtils.isEmpty(userList)) {
            return R.ok(0, true);
        }

        boolean bool = monthlyIncome(totalMoney, userList);
        if (!bool) {
            return R.ok(1, false);
        }
        return R.ok(0, true);

    }

    /**
     * 获取用户的商机信息
     *
     * @param userId 用户id
     * @return
     */
    @Override
    public List<User> getByList(String userId) {
        List<User> list = new ArrayList<>();
        User user = usersMapper.getByUserId(userId);
        getUser(list, userId);
        //去除本身
        list.remove(user);
        return list;
    }

    @XxlJob(value = "growthAwardDayTask")
    @Override
    public R<Boolean> progressPrizeCount() {
        log.info("shop-mall[]AccountServiceImpl[]performanceCount[]input.method");
        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);

        //1、判断当前月是否是业绩开始的第一个月
        List<TeamPerformance> teamPerformances = teamPerformanceMapper.getByBeforeTime(yearMonth);

        //获取当月所有人业绩总额
        Double totalMoney = teamPerformanceMapper.countByTime(yearMonth);
        double number = Math.floor(totalMoney / 3980 / 12);
        //2、集合为空 是业绩开始的第一个月
        if (CollectionUtils.isEmpty(teamPerformances)) {
            if (number != 0) {
                //获取月业绩前20用户
                List<TeamPerformance> list = teamPerformanceMapper.userTwenty(yearMonth);
                if (CollectionUtils.isEmpty(list)) {
                    return R.ok(0, true);
                }

                //获取20名进步最大的月业绩和
                Double twentyTotal = teamPerformanceMapper.countTwenty(yearMonth);
                List<AccountInfo> accountInfoList = new ArrayList<>();
                List<TradeRecord> tradeRecordList = new ArrayList<>();
                for (TeamPerformance user : list) {
                    //用户是普通用户的，不计算最大进步奖
                    if (user.getUserLevel().equals(UserLevelEnum.NORMAL_USER.getCode())) {
                        continue;
                    }

                    //获取最大进步奖 百分比
                    MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(user.getUserLevel(), 3);

                    //计算收益
                    double userTeamPerformance = user.getMonthTeamPerformance().doubleValue();
                    double percent = memberPercent.getPercent().doubleValue();

                    double income = Math.floor(number * 3980 * percent * userTeamPerformance / twentyTotal);

                    //获取账户信息
                    AccountInfo accountInfo = accountMapper.getByUserId(user.getUserId());

                    //用户月收益  =其他收益+最大进步奖收益
                    BigDecimal earningsMonth = accountInfo.getEarningsMonth().add(BigDecimal.valueOf(income));
                    accountInfo.setEarningsMonth(earningsMonth);

                    //用户总收益 =其他总收益 + earningsMonth
                    BigDecimal earningsTotal = accountInfo.getEarningsTotal().add(BigDecimal.valueOf(income));
                    accountInfo.setEarningsTotal(earningsTotal);
                    accountInfoList.add(accountInfo);

                    //添加交易流水记录
                    TradeRecord tradeRecord = new TradeRecord();
                    tradeRecord.setUserId(user.getUserId());
                    tradeRecord.setTradeType(TradeRecordEnum.PROGRESS_PRIZE.getCode());
                    tradeRecord.setTradeNo(null);
                    tradeRecord.setStatus(TradeStatusEnum.NO_SETTLE_ACCOUNTS.getCode());
                    tradeRecord.setMoney(BigDecimal.valueOf(income));
                    tradeRecordList.add(tradeRecord);
                }

                //更新账户信息，添加交易流水记录
                boolean bool = accountManager.updateAccountAddRecord(accountInfoList, tradeRecordList);
                if (!bool) {
                    return R.ok(1, false);
                }
            }
            return R.ok(0, true);
        }

        //3、集合不为空 不是业绩开始的第一个月
        //获取用户列表
        List<User> userList = usersMapper.findAll();
        if (CollectionUtils.isEmpty(userList)) {
            return R.ok(0, true);
        }

        //4计算每个用户本月的业绩增长率
        List<TeamPerformanceSortVo> listVo = progressPrizeStatistics(userList);

        //5、如果集合大于20 ，取前二十名，小于，取全部
        if (listVo.size() >= TWENTY) {
            //取排名前20的
            listVo.subList(0, TWENTY);

            //计算前20的总业绩
            double total = listVo.stream().mapToDouble(TeamPerformanceSortVo::getMonthPerformance).sum();
            totalPerformanceIncome(listVo, number, total);
            return R.ok(0, true);
        } else {
            //计算用户数少于20的总业绩
            double total = listVo.stream().mapToDouble(TeamPerformanceSortVo::getMonthPerformance).sum();
            totalPerformanceIncome(listVo, number, total);
        }
        return R.ok(0, true);
    }


    public void getUser(List<User> list, String userId) {
        User user = usersMapper.getByUserId(userId);
        list.add(user);
        if (null != user && !StringUtils.isBlank(user.getBeInvitedCode())) {
            User userInfo = usersMapper.getByBeInvitedCode(user.getBeInvitedCode());
            if (null != userInfo) {
                getUser(list, userInfo.getUserId());
            }
        }
    }

    /**
     * 如果会员等级是黄金以上，计算月度收益
     *
     * @param totalMoney 月金额总额
     * @param userList   用户列表
     * @return
     */
    private boolean monthlyIncome(BigDecimal totalMoney, List<User> userList) {
        double total = totalMoney.doubleValue();
        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);

        //当月所有人订单成交总金额/3980/12  向下整
        double moneyMonth = Math.floor(total / 3980 / 12);
        List<AccountInfo> accountInfoList = new ArrayList<>();
        List<TradeRecord> tradeRecordList = new ArrayList<>();

        //获取上月未分配完的奖金
        double lastMonthAward;
        Date last = DateUtil.getLastMonth(new Date());
        String yearMonthTime = DateUtil.convertDateToStr(last, PATTERN);
        MonthManure monthManure = monthManureMapper.queryByTime(yearMonthTime);
        if (null == monthManure) {
            lastMonthAward = 0.00;
        } else {
            lastMonthAward = monthManure.getManureAward();
        }
        for (User user : userList) {
            if (moneyMonth != 0) {
                //获取等级优惠百分比
                MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(user.getUserLevel(), 2);
                if (null != memberPercent) {
                    //获取本月团队总金额
                    TeamPerformance teamPerformance = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), yearMonth);

                    //获取该等级团队总金额
                    Double teamTotal = teamPerformanceMapper.countTeamMoney(user.getUserLevel(), yearMonth);
                    if (null != teamPerformance && null != teamTotal) {
                        double month = teamPerformance.getMonthTeamPerformance().doubleValue();
                        double percent = memberPercent.getPercent().doubleValue();
                        //计算收益
                        double income = (moneyMonth + lastMonthAward) * 3980 * percent * month / teamTotal;

                        //获取账户信息
                        AccountInfo accountInfo = accountMapper.getByUserId(user.getUserId());
                        BigDecimal bigDecimal = new BigDecimal(income);
                        BigDecimal performanceMonth = accountInfo.getEarningsMonth().add(bigDecimal);
                        accountInfo.setEarningsMonth(performanceMonth);

                        BigDecimal performanceTotal = accountInfo.getEarningsMonth().add(bigDecimal);
                        accountInfo.setEarningsMonth(performanceTotal);
                        accountInfoList.add(accountInfo);


                        //5、添加交易流水记录
                        TradeRecord tradeRecord = new TradeRecord();
                        tradeRecord.setUserId(user.getUserId());
                        tradeRecord.setTradeType(TradeRecordEnum.MONTHLY_FERTILIZER.getCode());
                        tradeRecord.setTradeNo(null);
                        tradeRecord.setStatus(TradeStatusEnum.NO_SETTLE_ACCOUNTS.getCode());
                        tradeRecordList.add(tradeRecord);
                    }
                }
            }
        }
        //更新账户信息，新增交易流水记录
        return accountManager.updateAccountAddRecord(accountInfoList, tradeRecordList);
    }

    /**
     * 统计每个用户本月业绩增长率
     *
     * @param userList 用户列表
     * @return 每个用户本月业绩增长率
     */
    private List<TeamPerformanceSortVo> progressPrizeStatistics(List<User> userList) {
        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);

        //遍历 计算业绩增长率
        List<TeamPerformanceSortVo> listVo = new ArrayList<>();
        for (User user : userList) {
            TeamPerformanceSortVo teamVo = new TeamPerformanceSortVo();
            double growthRate;
            //获取当月业绩
            TeamPerformance teamPerformance = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), yearMonth);

            //获取上月业绩
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date());
            cal.add(Calendar.MONTH, -1);
            Date lastDate = cal.getTime();
            String lastMonthTime = DateUtil.convertDateToStr(lastDate, PATTERN);
            TeamPerformance lastMonth = teamPerformanceMapper.getByUserIdAndTime(user.getUserId(), lastMonthTime);
            if (null == teamPerformance || null == lastMonth) {
                growthRate = 0.00;
            } else {
                double month = teamPerformance.getMonthTeamPerformance().doubleValue();
                double last = lastMonth.getMonthTeamPerformance().doubleValue();
                if (last >= month) {
                    growthRate = 0.00;
                } else {
                    growthRate = (month - last) / month;
                }
            }
            teamVo.setGrowthRate(growthRate);
            double monthPerformance;
            assert teamPerformance != null;
            if (null == teamPerformance.getMonthTeamPerformance()) {
                monthPerformance = 0.00;
            } else {
                monthPerformance = teamPerformance.getMonthTeamPerformance().doubleValue();
            }
            teamVo.setMonthPerformance(monthPerformance);
            teamVo.setUserId(user.getUserId());
            teamVo.setTeamPerformance(teamPerformance);
            listVo.add(teamVo);
        }

        //对集合进行排序
        listVo.sort(Comparator.comparing(TeamPerformanceSortVo::getGrowthRate).reversed());
        return listVo;
    }

    /**
     * 统计用户最大进步奖收益
     *
     * @param listVo 用户增长率列表
     * @param number 个人业绩
     * @param total  总业绩
     */
    private void totalPerformanceIncome(List<TeamPerformanceSortVo> listVo, double number, double total) {
        //要更新的账户列表
        List<AccountInfo> updateAccountList = new ArrayList<>();

        BigDecimal sum = new BigDecimal(0);
        for (int i = 0; i < listVo.size(); i++) {
            String userId = listVo.get(i).getTeamPerformance().getUserId();
            Integer userLevel = listVo.get(i).getTeamPerformance().getUserLevel();
            //获取最大进步奖 百分比
            MemberPercent memberPercent = memberPercentMapper.getByLevelAndType(userLevel, 3);

            //计算收益
            double userTeamPerformance = listVo.get(i).getTeamPerformance().getMonthTeamPerformance().doubleValue();
            double percent = memberPercent.getPercent().doubleValue();

            double income = number * 3980 * percent * userTeamPerformance / total;

            //获取账户信息
            AccountInfo accountInfo = accountMapper.getByUserId(userId);
            BigDecimal bigDecimal;
            if (i == listVo.size() - 1) {
                bigDecimal = new BigDecimal(total).subtract(sum);
            } else {
                bigDecimal = accountInfo.getEarningsMonth().add(new BigDecimal(income));
                sum = sum.add(new BigDecimal(income));
            }
            accountInfo.setEarningsMonth(bigDecimal);
            updateAccountList.add(accountInfo);
        }

        //判断本月是否有最大进步奖数据 无，新增  有，修改或删除
        Date date = new Date();
        String yearMonth = DateUtil.convertDateToStr(date, PATTERN);
        List<ProgressPrize> prizes = progressPrizeMapper.getByYearMonth(yearMonth);

        //修改或保存最大进步奖信息
        accountManager.updateOrSavePrize(listVo, updateAccountList, prizes);
    }
}
