package cn.wisenergy.web.aspect;

import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * AOP日志
 */
@Aspect
@Component
@Slf4j
public class WebLogAspect {

    /**
     * 两个..代表所有子目录，最后括号里的两个..代表所有参数
     */
    @Pointcut("execution( * cn.wisenergy.web.admin.controller..*.*(..))")
    public void logPointCut() {
    }

    @Before("logPointCut()")
    public void doBefore(JoinPoint joinPoint) {
        // 接收到请求，记录请求内容
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();

        // 记录下请求内容
        log.info(" ___________________________________________________________");
        log.info("| 请求地址 : {}", request.getRequestURL().toString());
        log.info("| 请求方式 : {}", request.getMethod());
        // 获取执行的方法全路径
        String methodName = joinPoint.getSignature().getName();
        log.info("| 执行方法 : {}.{}", joinPoint.getSignature().getDeclaringTypeName(), methodName);
    }

    /**
     * returning的值和doAfterReturning的参数名一致
     */
    @AfterReturning(returning = "ret", pointcut = "logPointCut()")
    public void doAfterReturning(Object ret) {
        // 处理完请求，返回内容
        log.info("| 返回值 : {}", JSON.toJSONString(ret));
        log.info(" ———————————————————————————————————————————");
    }

    @Around("logPointCut()")
    public Object doAround(ProceedingJoinPoint pjp) throws Throwable {
        long startTime = System.currentTimeMillis();
        // ob 为方法的返回值
        Object ob = pjp.proceed();
        log.info("| 耗时 : {}", (System.currentTimeMillis() - startTime));
        return ob;
    }
}
