package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.utils.R;
import cn.wisenergy.common.utils.StringUtil;
import cn.wisenergy.mapper.*;
import cn.wisenergy.model.app.*;
import cn.wisenergy.model.enums.SchemeTypeEnums;
import cn.wisenergy.model.enums.StudentClassEnum;
import cn.wisenergy.model.enums.StudentType;
import cn.wisenergy.model.vo.SchemeQueryVo;
import cn.wisenergy.model.vo.VolunteerVo;
import cn.wisenergy.service.app.SchemeService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author 86187
 * @ Description: 方案接口实现
 * @ Author     : 86187
 * @ Date       : 2021/1/13 15:04
 */
@Service
@Slf4j
public class SchemeServiceImpl extends ServiceImpl<SchemeMapper, SchemeInfo> implements SchemeService {
    @Autowired
    private UsersMapper usersMapper;

    @Autowired
    private ScoreInfoMapper scoreInfoMapper;

    @Autowired
    private VolunteerMapper volunteerMapper;

    @Autowired
    private SchemeRecordMapper schemeRecordMapper;

    @Autowired
    private UserVolunteerMapper userVolunteerMapper;

    @Autowired
    private AdmissionRuleMapper admissionRuleMapper;

    @Autowired
    private ProfessionMapper professionMapper;

    @Override
    public R<VolunteerVo> getList(SchemeQueryVo queryVo) {
        log.info("volunteer-service[]SchemeServiceImpl[]getList[]input.param.queryVo:" + queryVo);
        if (null == queryVo || null == queryVo.getUserId() || null == queryVo.getCultureGrade()) {
            return R.error("入参为空!");
        }

        if (StringUtils.isBlank(queryVo.getProfessionIds())) {
            return R.error("未选择专业!");
        }

        //1、根据userId获取用户信息
        User user = usersMapper.selectById(queryVo.getUserId());
        if (null == user) {
            return R.error("用户信息不存在!");
        }

        //2、获取考生成绩
        QueryWrapper<ScoreInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", queryVo.getUserId());
        ScoreInfo scoreInfo = scoreInfoMapper.selectOne(queryWrapper);
        if (null == scoreInfo) {
            return R.error("无考生成绩!");
        }


        //3、根据筛选方案算法，筛选方案
        //副科名称
        List<String> className = getClassName(scoreInfo);
        if (CollectionUtils.isEmpty(className)) {
            return R.error("考生无副科成绩!");
        }

        //选取专业名称
        List<String> professionName = getProfessionName(queryVo);
        if (CollectionUtils.isEmpty(professionName)) {
            return R.error("未选择专业!");
        }

        //(1)判断考生类型 文化课考生
        List<Volunteer> list = new ArrayList<>();
        VolunteerVo volunteerVo = new VolunteerVo();
        if (StudentType.CULTURE_STUDENT.getCode().equals(user.getExamType())) {
            //获取文化课本科一批规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
            if (null == firstRule) {
                return R.error("无文化课本科录取规则!");
            }

            //获取文化课专科一批规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());
            if (null == secondRule) {
                return R.error("无文化课专科录取规则!");
            }

            //1).判断文化课的分数是否大于等于最大文化分
            //本科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double firstCulture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
            double cultureMax = Double.parseDouble(firstRule.getCultureMax().toString());
            if (firstCulture >= cultureMax) {
                volunteerVo.setUserId(user.getId());
                volunteerVo.setVolunteers(list);
                return R.ok("恭喜您，因您的成绩突出，请联系官方免费领取VIP一对一报考方案指导！", volunteerVo);
            }

            //2).判断文化课的分数是否小于最小报考分数
            //专科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
            double cultureMin = Double.parseDouble(secondRule.getCultureMin().toString());
            if (secondCulture < cultureMin) {
                return R.error("很抱歉，您的成绩不满足报考条件，无法查询!");
            }

            list = getCultureList(scoreInfo, firstRule, secondRule, className, professionName, firstCulture, secondCulture);
        }

        //(2)美术考生
        if (StudentType.ART_STUDENT.getCode().equals(user.getExamType())) {
            //获取美术本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            if (null == firstRule) {
                return R.error("无美术本科录取规则!");
            }

            //获取美术专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
            if (null == secondRule) {
                return R.error("无美术专科录取规则!");
            }

            //判断考生是否有报取资格
            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());
            if (secondMajor < secondRule.getProfessionMin()) {
                return R.error("很抱歉，您的成绩不满足报考条件，无法查询!");
            }
            list = getArtsList(scoreInfo, firstRule, secondRule, className, professionName);
        }

        //(3)文学编导考生
        if (StudentType.LITERATURE_STUDENT.getCode().equals(user.getExamType())) {
            //获取文学编导本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            if (null == firstRule) {
                return R.error("无文学编导本科录取规则!");
            }

            //获取文学编导专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            if (null == secondRule) {
                return R.error("无文学编导专科录取规则!");
            }

            //判断考生是否有报取资格
            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());
            if (secondMajor < secondRule.getProfessionMin()) {
                return R.error("很抱歉，您的专业成绩不满足报考条件，无法查询!");
            }
            list = getLiteratureList(scoreInfo, firstRule, secondRule, className, professionName);
        }

        //(4)体育考生
        if (StudentType.SPORTS_STUDENT.getCode().equals(user.getExamType())) {
            //获取体育本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            if (null == firstRule) {
                return R.error("无体育本科录取规则!");
            }

            //获取体育专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode());
            if (null == secondRule) {
                return R.error("无体育专科录取规则!");
            }

            //判断考生是否有报取资格
            //综合成绩差值
            double value = secondRule.getCurrentYearCulture() - secondRule.getCultureMin();
            double culture = Double.parseDouble(queryVo.getCultureGrade());
            double major = Double.parseDouble(queryVo.getMajorGrade());

            //计算综合成绩
            double total = major * 750 / 100 * 0.7 + culture * 0.3 - value;
            if (total < secondRule.getProfessionMin()) {
                return R.error("很抱歉，您的成绩不满足报考条件，无法查询!");
            }
            list = getSportsList(total, firstRule, secondRule, className, professionName);
        }

        //4保存用户方案记录、志愿之间的关联关系
        boolean bool = saveUserVolunteer(user, list, scoreInfo);
        if (!bool) {
            return R.error("保存方案查询记录失败!");
        }

        volunteerVo.setUserId(user.getId());
        volunteerVo.setVolunteers(list);
        return R.ok(volunteerVo);
    }

    @Override
    public R<Boolean> deleteById(Integer schemeId) {
        log.info("volunteer-service[]SchemeServiceImpl[]deleteById[]input.param.schemeId:" + schemeId);
        if (null == schemeId) {
            return R.error("入参为空!");
        }
        //根据方案id,获取方案信息
        QueryWrapper<SchemeInfo> query = new QueryWrapper<>();
        query.eq("id", schemeId);
        query.eq("is_delete", 0);
        SchemeInfo schemeInfo = baseMapper.selectOne(query);
        if (null == schemeInfo) {
            return R.error("方案信息不存在!");
        }

        //todo 关联删除，需要加事务
        //更新方案状态为 ：1 删除
        schemeInfo.setIsDelete(1);
        int sum = baseMapper.updateById(schemeInfo);
        if (sum == 0) {
            return R.ok(1, false);
        }

        //根据方案id,获取志愿信息
        QueryWrapper<Volunteer> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("scheme_id", schemeId);
        queryWrapper.eq("is_delete", 0);
        List<Volunteer> list = volunteerMapper.selectList(queryWrapper);
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(0, true);
        }

        int count = volunteerMapper.updateBySchemeId(schemeId);
        if (list.size() != count) {
            return R.ok(1, false);
        }
        return R.ok(0, true);
    }


    /**
     * 保存方案查询记录和关联关系
     *
     * @param user 用户信息
     * @param list     志愿信息
     * @return true 成功  false 失败
     */
    @Transactional
    public boolean saveUserVolunteer(User user, List<Volunteer> list, ScoreInfo scoreInfo) {
        //1、保存方案查询记录
        SchemeQueryRecord schemeQueryRecord = new SchemeQueryRecord();
        schemeQueryRecord.setMajorName("");
        schemeQueryRecord.setStudentType(user.getExamType());
        schemeQueryRecord.setUserId(user.getId());
        schemeQueryRecord.setIsDelete(0);
        double score = scoreInfo.getCultureGrade() + scoreInfo.getMajorGrade();
        schemeQueryRecord.setScore(String.valueOf(score));

        List<Integer> ids = list.stream().map(Volunteer::getId).collect(Collectors.toList());
        //保存方案查询记录
        SchemeQueryRecord count = schemeRecordMapper.add(schemeQueryRecord);
        if (null == count) {
            return false;
        }

        List<UserVolunteer> volunteerList = new ArrayList<>();
        for (Integer id : ids) {
            UserVolunteer userVolunteer = new UserVolunteer();
            userVolunteer.setSchemeRecordId(count.getId());
            userVolunteer.setUserId(user.getId());
            userVolunteer.setVolunteerId(id);
            volunteerList.add(userVolunteer);
        }

        int save = userVolunteerMapper.creates(volunteerList);
        if (volunteerList.size() != save) {
            return false;
        }
        return true;
    }

    /**
     * 获取文化生志愿列表
     *
     * @param scoreInfo       成绩
     * @param firstRule       录取规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 文化生志愿列表
     */
    private List<Volunteer> getCultureList(ScoreInfo scoreInfo, AdmissionRule firstRule, AdmissionRule secondRule,
                                           List<String> classNames, List<String> professionNames,
                                           double firstCulture, double secondCulture) {
        double cultureMax = firstRule.getCultureMax();
        double cultureMin = firstRule.getCultureMin();
        double downMark = firstRule.getDownMark();

        //考生的分数在本科一批中
        List<Volunteer> list = new ArrayList<>();
        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);
        if (firstCulture < cultureMax && firstCulture > cultureMin) {
            //判断分数是否小于对低分与向下浮动分数之和
            double upMark = firstRule.getUpMark();
            double upGrade = firstCulture + upMark;
            map.put("upGrade", upGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());

            double downGrade;
            if (firstCulture < cultureMin + downMark) {
                downGrade = cultureMin;
                map.put("downGrade", downGrade);
                list = volunteerMapper.getVolunteerList(map);
                //如果数据不满足最大数量，取不限专业重最低分补充
                if (list.size() < firstRule.getNumber()) {
                    List<Integer> volunteerIds = list.stream().map(Volunteer::getId).collect(Collectors.toList());
                    int total = firstRule.getNumber() - list.size();
                    List<Volunteer> fillList = fillUpVolunteer(volunteerIds, total, upGrade, downGrade);
                    list.addAll(fillList);
                }
            } else {
                downGrade = firstCulture - firstRule.getDownMark();
                map.put("downGrade", downGrade);
                list = volunteerMapper.getVolunteerList(map);
            }
        } else {
            //考生的分数是在专科一批中
            //判断分数是否小于对低分与向下浮动分数之和
            double upMark = secondRule.getUpMark();
            double upGrade = secondCulture + upMark;
            if (upGrade > secondRule.getCultureMax()) {
                upGrade = secondRule.getCultureMax();
            }

            map.put("upGrade", upGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());

            double downGrade;
            if (secondCulture < cultureMin + downMark) {
                downGrade = cultureMin;
                map.put("downGrade", downGrade);
                list = volunteerMapper.getVolunteerList(map);

                //如果数据不满足最大数量，取不限专业最低分补充
                if (list.size() < firstRule.getNumber()) {
                    List<Integer> volunteerIds = list.stream().map(Volunteer::getId).collect(Collectors.toList());
                    int total = firstRule.getNumber() - list.size();
                    List<Volunteer> fillList = fillUpVolunteer(volunteerIds, total, upGrade, downGrade);
                    list.addAll(fillList);
                }
            } else {
                downGrade = secondCulture - secondRule.getDownMark();
                map.put("downGrade", downGrade);
                list = volunteerMapper.getVolunteerList(map);
            }
        }
        return list;
    }

    /**
     * 获取美术生志愿方案
     *
     * @param scoreInfo       考生成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getArtsList(ScoreInfo scoreInfo, AdmissionRule firstRule, AdmissionRule secondRule,
                                        List<String> classNames, List<String> professionNames) {
        //本科类比分 = 本年录取分 -去年录取分 + 考生分数
        double culture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + scoreInfo.getCultureGrade();
        double major = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + scoreInfo.getMajorGrade();
        //计算综合成绩
        double total = major * 750 / 300 * 0.7 + culture * 0.3;

        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);

        //1、考生是否能报取美术本科
        List<Volunteer> list = new ArrayList<>();
        if (culture >= firstRule.getCultureMin() && major >= firstRule.getProfessionMin()) {
            //浮动分数
            double upGrade = total + firstRule.getUpMark();
            double downGrade = total + firstRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            list = volunteerMapper.getVolunteerList(map);

        }

        //专科科类比分 = 本年录取分 -去年录取分 + 考生分数
        double secondRuleCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() + scoreInfo.getCultureGrade();
        double secondRuleMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() + scoreInfo.getMajorGrade();
        //计算综合成绩
        double secondRuleTotal = secondRuleMajor * 750 / 300 * 0.7 + secondRuleCulture * 0.3;
        //2、考生是否能报取美术专科
        if (culture >= secondRule.getCultureMin() && major >= secondRule.getProfessionMin()) {
            //浮动分数
            double upGrade = secondRuleTotal + secondRule.getUpMark();
            double downGrade = secondRuleTotal + secondRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
            list = volunteerMapper.getVolunteerList(map);
        }

        return list;
    }

    /**
     * 获取文学编导生志愿方案
     *
     * @param scoreInfo       考生成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getLiteratureList(ScoreInfo scoreInfo, AdmissionRule firstRule, AdmissionRule secondRule,
                                              List<String> classNames, List<String> professionNames) {
        double culture = scoreInfo.getCultureGrade();
        double major = scoreInfo.getMajorGrade();
        //计算综合成绩
        double total = major * 750 / 300 * 0.7 + culture * 0.3;

        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);

        //1、考生是否能报取美术本科
        List<Volunteer> list = new ArrayList<>();
        if (culture >= firstRule.getCultureMin() && major >= firstRule.getProfessionMin()) {
            //浮动分数
            double upGrade = total + firstRule.getUpMark();
            double downGrade = total + firstRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            list = volunteerMapper.getVolunteerList(map);

        }

        //2、考生是否能报取美术专科
        if (culture >= secondRule.getCultureMin() && major >= secondRule.getProfessionMin()) {
            //浮动分数
            double upGrade = total + secondRule.getUpMark();
            double downGrade = total + secondRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            list = volunteerMapper.getVolunteerList(map);
        }

        return list;
    }

    /**
     * 获取体育生生志愿方案
     *
     * @param total           考生综合成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getSportsList(double total, AdmissionRule firstRule, AdmissionRule secondRule,
                                          List<String> classNames, List<String> professionNames) {
        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);

        //1、考生是否能报取体育本科
        List<Volunteer> list = new ArrayList<>();
        if (total >= firstRule.getCultureMin()) {
            //浮动分数
            double upGrade = total + firstRule.getUpMark();
            double downGrade = total + firstRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            list = volunteerMapper.getVolunteerList(map);

        }

        //2、考生是否能报取体育专科
        if (total >= secondRule.getCultureMin()) {
            //浮动分数
            double upGrade = total + secondRule.getUpMark();
            double downGrade = total + secondRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode());
            list = volunteerMapper.getVolunteerList(map);
        }

        return list;
    }

    /**
     * 获取副科的名称
     *
     * @param scoreInfo 成绩信息
     * @return 副科列表
     */
    private List<String> getClassName(ScoreInfo scoreInfo) {
        List<String> name = new ArrayList<>();
        name.add(StudentClassEnum.UNLIMITED.getDesc());
        if (scoreInfo.getPhysicsGrade() > 0) {
            name.add(StudentClassEnum.PHYSICS_GRADE.getDesc());
        }

        if (scoreInfo.getChemistryGrade() > 0) {
            name.add(StudentClassEnum.CHEMISTRY_GRADE.getDesc());
        }

        if (scoreInfo.getBiologyGrade() > 0) {
            name.add(StudentClassEnum.BIOLOGY_GRADE.getDesc());
        }

        if (scoreInfo.getHistoryGrade() > 0) {
            name.add(StudentClassEnum.HISTORY_GRADE.getDesc());
        }

        if (scoreInfo.getGeographyGrade() > 0) {
            name.add(StudentClassEnum.GEOGRAPHY_GRADE.getDesc());
        }

        if (scoreInfo.getPoliticsGrade() > 0) {
            name.add(StudentClassEnum.POLITICS_GRADE.getDesc());
        }

        return name;
    }

    /**
     * 获取专业名称
     *
     * @param schemeQueryVo 专业参数
     * @return 专业名称列表
     */
    private List<String> getProfessionName(SchemeQueryVo schemeQueryVo) {
        //把字符专ids业转为数组
        List<Integer> professionIds = StringUtil.strToArray(schemeQueryVo.getProfessionIds());
        return professionMapper.getNameByIds(professionIds);
    }

    /**
     * 补充不足数量的志愿数
     *
     * @param ids    已有志愿ids
     * @param number 差的数量
     * @param max    最高分
     * @param min    最低分
     * @return 集合
     */
    private List<Volunteer> fillUpVolunteer(List<Integer> ids, Integer number, double max, double min) {
        Map<String, Object> map = new HashMap<>(16);
        map.put("volunteerIds", ids);
        map.put("number", number);
        map.put("upGrade", max);
        map.put("downGrade", min);
        return volunteerMapper.getFillList(map);
    }
}
