package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.constant.CommonAttributes;
import cn.wisenergy.common.utils.R;
import cn.wisenergy.common.utils.StringUtil;
import cn.wisenergy.mapper.*;
import cn.wisenergy.model.app.*;
import cn.wisenergy.model.enums.SchemeTypeEnums;
import cn.wisenergy.model.enums.StudentClassEnum;
import cn.wisenergy.model.enums.StudentType;
import cn.wisenergy.model.vo.*;
import cn.wisenergy.service.app.SchemeService;
import cn.wisenergy.service.app.UserVolunteerService;
import cn.wisenergy.service.common.VolunteerManager;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageInfo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author 86187
 * @ Description: 方案接口实现
 * @ Author     : 86187
 * @ Date       : 2021/1/13 15:04
 */
@Service
@Slf4j
public class SchemeServiceImpl extends ServiceImpl<SchemeMapper, SchemeInfo> implements SchemeService {
    @Autowired
    private UsersMapper usersMapper;

    @Autowired
    private SchemeMapper schemeMapper;

    @Autowired
    private ScoreInfoMapper scoreInfoMapper;

    @Autowired
    private VolunteerMapper volunteerMapper;

    @Autowired
    private SchemeRecordMapper schemeRecordMapper;

    @Autowired
    private UserVolunteerMapper userVolunteerMapper;

    @Autowired
    private AdmissionRuleMapper admissionRuleMapper;

    @Autowired
    private ProfessionMapper professionMapper;

    @Autowired
    private UserVolunteerService userVolunteerService;

    @Autowired
    private VolunteerManager volunteerManager;

    @Override
    public R<VolunteerVo> getList(SchemeVo queryVo) {
        log.info("volunteer-service[]SchemeServiceImpl[]getList[]input.param.queryVo:" + queryVo);
        if (null == queryVo || null == queryVo.getUserId() || null == queryVo.getCultureGrade()) {
            return R.error("入参为空!");
        }

        if (StringUtils.isBlank(queryVo.getProfessionIds())) {
            return R.error("未选择专业!");
        }

        //1、根据userId获取用户信息
        User user = usersMapper.selectById(queryVo.getUserId());
        if (null == user) {
            return R.error("用户信息不存在!");
        }

        //2、获取考生成绩
        QueryWrapper<ScoreInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", queryVo.getUserId());
        ScoreInfo scoreInfo = scoreInfoMapper.selectOne(queryWrapper);
        if (null == scoreInfo) {
            return R.error("无考生成绩!");
        }

        //3、根据筛选方案算法，筛选方案
        //副科名称
        List<String> className = getClassName(scoreInfo);
        if (CollectionUtils.isEmpty(className)) {
            return R.error("考生无副科成绩!");
        }

        //选取专业名称
        List<String> professionName = getProfessionName(queryVo);
        if (CollectionUtils.isEmpty(professionName)) {
            return R.error("未选择专业!");
        }

        //(1)判断考生类型 文化课考生
        //初始化返回数据总条数
        int totalResult = 0;
        List<Volunteer> list = new ArrayList<>();
        VolunteerVo volunteerVo = new VolunteerVo();
        if (StudentType.CULTURE_STUDENT.getCode().equals(user.getExamType())) {
            //获取文化课本科一批规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
            if (null == firstRule) {
                return R.error("无文化课本科录取规则!");
            }

            //获取文化课专科一批规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());
            if (null == secondRule) {
                return R.error("无文化课专科录取规则!");
            }

            //1).判断文化课的分数是否大于等于最大文化分
            //本科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double firstCulture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
            double cultureMax = Double.parseDouble(firstRule.getCultureMax().toString());
            if (firstCulture >= cultureMax) {
                volunteerVo.setUserId(user.getId());
                volunteerVo.setVolunteers(list);
                return R.ok("恭喜您，因您的成绩突出，请联系官方免费领取VIP一对一报考方案指导！", volunteerVo);
            }

            //2).判断文化课的分数是否小于最小报考分数
            //专科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
            double cultureMin = Double.parseDouble(secondRule.getCultureMin().toString());
            if (secondCulture < cultureMin) {
                return R.error("很抱歉，您的成绩不满足报考条件，无法查询!");
            }

            totalResult = firstRule.getNumber();
            list = getCultureList(scoreInfo, firstRule, secondRule, className, professionName, firstCulture, secondCulture);
        }

        //(2)美术考生
        if (StudentType.ART_STUDENT.getCode().equals(user.getExamType())) {
            //获取美术本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            if (null == firstRule) {
                return R.error("无美术本科录取规则!");
            }

            //获取美术专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
            if (null == secondRule) {
                return R.error("无美术专科录取规则!");
            }

            //判断考生是否有报取资格
            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() +
                    Double.parseDouble(queryVo.getMajorGrade());
            double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() +
                    Double.parseDouble(queryVo.getMajorGrade());
            if (secondCulture < secondRule.getCultureMin() && secondMajor < secondRule.getProfessionMin()) {
                return R.error("很抱歉，您的成绩不满足报考条件，无法查询!");
            }

            totalResult = firstRule.getNumber();
            list = getArtsList(queryVo, firstRule, secondRule, className, professionName);
        }

        //(3)文学编导考生
        if (StudentType.LITERATURE_STUDENT.getCode().equals(user.getExamType())) {
            //获取文学编导本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            if (null == firstRule) {
                return R.error("无文学编导本科录取规则!");
            }

            //获取文学编导专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            if (null == secondRule) {
                return R.error("无文学编导专科录取规则!");
            }

            //判断考生是否有报取资格
            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() +
                    Double.parseDouble(queryVo.getCultureGrade());
            double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() +
                    Double.parseDouble(queryVo.getMajorGrade());
            if (secondCulture < secondRule.getCultureMin() || secondMajor < secondRule.getProfessionMin()) {
                return R.error("很抱歉，您的专业成绩不满足报考条件，无法查询!");
            }

            totalResult = firstRule.getNumber();
            list = getLiteratureList(queryVo, firstRule, secondRule, className, professionName);
        }

        //(4)体育考生
        if (StudentType.SPORTS_STUDENT.getCode().equals(user.getExamType())) {
            //获取体育本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            if (null == firstRule) {
                return R.error("无体育本科录取规则!");
            }

            //获取体育专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode());
            if (null == secondRule) {
                return R.error("无体育专科录取规则!");
            }

            //判断考生是否有报取资格
            //综合成绩差值
            double value = secondRule.getCurrentYearCulture() - secondRule.getCultureMin();
            double culture = Double.parseDouble(queryVo.getCultureGrade());
            double major = Double.parseDouble(queryVo.getMajorGrade());

            //计算综合成绩
            double total = major * 750 / 100 * 0.7 + culture * 0.3 - value;
            if (total < secondRule.getCultureMin()) {
                return R.error("很抱歉，您的成绩不满足报考条件，无法查询!");
            }

            totalResult = firstRule.getNumber();
            list = getSportsList(total, firstRule, secondRule, className, professionName);
        }

        //4、判断返回的结果数据是否大于totalResult
        if (list.size() > totalResult) {
            list = getVolunteerList(list, totalResult);
        }

        //排序
        list.sort(Comparator.comparing(Volunteer::getLowestMark).reversed());

        //5判断是否展示全部方案
        if (queryVo.getIsShowAll() == 0) {
            List<Volunteer> result;
            if (list.size() < 10) {
                result = list.subList(0, list.size());
            } else {
                result = list.subList(0, 10);
            }
            volunteerVo.setUserId(user.getId());
            //设置查询时间
            SetQueryTime(result);
            volunteerVo.setVolunteers(result);
            List<Volunteer> volunteers = volunteerVo.getVolunteers();
            //为空的最低位次用--代替
            for (Volunteer info : volunteers) {
                if (null == info.getLowestRank()) {
                    info.setLowestRank("--");
                }
            }
            return R.ok(volunteerVo);
        }

        //6保存用户方案记录、志愿之间的关联关系,扣减查询次数
        R<Integer> bool = volunteerManager.saveUserVolunteer(user, list, scoreInfo, queryVo.getBigMajorNames());
        if (null != bool && bool.getCode() == 500) {
            return R.error(bool.getMessage());
        }

        volunteerVo.setUserId(user.getId());
        assert bool != null;
        volunteerVo.setRecordId(bool.getData());

        //设置查询时间
        SetQueryTime(list);
        volunteerVo.setVolunteers(list);
        volunteerVo.setTotal(list.size());

        return R.ok(volunteerVo);
    }

    @Override
    public R<Boolean> deleteById(Integer schemeId) {
        log.info("volunteer-service[]SchemeServiceImpl[]deleteById[]input.param.schemeId:" + schemeId);
        if (null == schemeId) {
            return R.error("入参为空!");
        }
        //根据方案id,获取方案信息
        QueryWrapper<SchemeInfo> query = new QueryWrapper<>();
        query.eq("id", schemeId);
        query.eq("is_delete", 0);
        SchemeInfo schemeInfo = baseMapper.selectOne(query);
        if (null == schemeInfo) {
            return R.error("方案信息不存在!");
        }

        //更新方案状态为 ：1 删除
        schemeInfo.setIsDelete(1);

        //根据方案id,获取志愿信息
        QueryWrapper<Volunteer> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("scheme_id", schemeId);
        queryWrapper.eq("is_delete", 0);
        List<Volunteer> list = volunteerMapper.selectList(queryWrapper);

        boolean bool = volunteerManager.deleteSchemeAndVolunteer(schemeInfo, schemeId, list.size());
        return R.ok(bool);
    }

    @Override
    public R<PageInfo<SchemeInfo>> getSchemeList(SchemeQueryVo schemeVo) {
        log.info("volunteer-service[]SchemeServiceImpl[]getSchemeList[]input.param.schemeVo:" + schemeVo);
        if (null == schemeVo) {
            return R.error("入参为空!");
        }
        pageHandle(schemeVo);
        int total = schemeMapper.count();
        Map<String, Object> map = new HashMap<>(16);

        map.put("startNum", schemeVo.getStartNum());
        map.put("endNum", schemeVo.getEndNum());
        List<SchemeInfo> list = schemeMapper.getList(map);
        PageInfo<SchemeInfo> info = new PageInfo<>();
        info.setPageSize(schemeVo.getPageSize());
        info.setPageNum(schemeVo.getPageNo());
        info.setTotal(total);
        info.setList(list);
        return R.ok(info);
    }

    @Override
    public R<String> download(Integer schemeId) {
        log.info("volunteer-service[]SchemeServiceImpl[]download[]input.param.schemeId:" + schemeId);
        if (null == schemeId) {
            return R.error("入参为空!");
        }

        //获取方案信息
        SchemeInfo schemeInfo = schemeMapper.getById(schemeId);
        if (null == schemeInfo || StringUtils.isBlank(schemeInfo.getFileUrl())) {
            return R.error("未上传方案，无法下载!");
        }

        return R.ok(schemeInfo.getFileUrl());
    }

    @Override
    public List<Integer> ceishi(Integer id) {

        List<UserVolunteer> list = userVolunteerMapper.listById(id);
        List<Integer> ids = list.stream().map(UserVolunteer::getVolunteerId).collect(Collectors.toList());
        List<Integer> integerList = new ArrayList<>();
        List<Integer> result = new ArrayList<>();
        for (Integer vid : ids) {
            if (integerList.contains(vid)) {
                result.add(vid);
            } else {
                integerList.add(vid);
            }
        }
        return result;
    }

    @Override
    public R<VolunteerVo> getListVip(SchemeVipVo schemeVo) {
        log.info("volunteer-service[]getListVip[]getList[]input.param:" + schemeVo);
        if (null == schemeVo || null == schemeVo.getUserId()) {
            return R.error("入参为空!");
        }

        //综合成绩,地区
        Integer total = null == schemeVo.getTotal() ? null : schemeVo.getTotal();
        String local = null == schemeVo.getLocal() ? null : schemeVo.getLocal();

        //1、根据userId获取用户信息
        User user = usersMapper.selectById(schemeVo.getUserId());
        if (null == user) {
            return R.error("请先添加vip客户信息!");
        }

        //2、获取考生成绩
        QueryWrapper<ScoreInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", schemeVo.getUserId());
        ScoreInfo scoreInfo = scoreInfoMapper.selectOne(queryWrapper);
        if (null == scoreInfo) {
            return R.error("无vip客户成绩!");
        }

        //选取专业名称;
        List<Integer> professionIds = StringUtil.strToArray(schemeVo.getProfessionIds());
        List<String> nameByIds = professionMapper.getNameByIds(professionIds);
        if (CollectionUtils.isEmpty(nameByIds)) {
            return R.error("未选择专业!");
        }

        //(1)判断考生类型 文化课考生
        List<Volunteer> list = new ArrayList<>();
        VolunteerVo volunteerVo = new VolunteerVo();
        if (StudentType.CULTURE_STUDENT.getCode().equals(user.getExamType())) {
            //获取文化课本科一批规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
            if (null == firstRule) {
                return R.error("无文化课本科录取规则!");
            }

            //获取文化课专科一批规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());
            if (null == secondRule) {
                return R.error("无文化课专科录取规则!");
            }

            //本科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double firstCulture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(schemeVo.getCultureGrade());
            double cultureMax = Double.parseDouble(firstRule.getCultureMax().toString());

            //2).判断文化课的分数是否小于最小报考分数
            //专科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() + Double.parseDouble(schemeVo.getCultureGrade());
            double cultureMin = Double.parseDouble(secondRule.getCultureMin().toString());
            if (secondCulture < cultureMin) {
                return R.error("很抱歉，客户的成绩不满足报考条件，无法查询!");
            }

            list = getCultureVipList(firstRule, secondRule, total, local, nameByIds, firstCulture, secondCulture);
        }

        //(2)美术考生
        if (StudentType.ART_STUDENT.getCode().equals(user.getExamType())) {
            //获取美术本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            if (null == firstRule) {
                return R.error("无美术本科录取规则!");
            }

            //获取美术专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
            if (null == secondRule) {
                return R.error("无美术专科录取规则!");
            }

            //判断考生是否有报取资格
            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() +
                    Double.parseDouble(schemeVo.getMajorGrade());
            double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() +
                    Double.parseDouble(schemeVo.getMajorGrade());
            if (secondCulture < secondRule.getCultureMin() && secondMajor < secondRule.getProfessionMin()) {
                return R.error("很抱歉，客户的成绩不满足报考条件，无法查询!");
            }
            list = getArtsVipList(schemeVo, firstRule, secondRule, total, local, nameByIds);
        }

        //(3)文学编导考生
        if (StudentType.LITERATURE_STUDENT.getCode().equals(user.getExamType())) {
            //获取文学编导本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            if (null == firstRule) {
                return R.error("无文学编导本科录取规则!");
            }

            //获取文学编导专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            if (null == secondRule) {
                return R.error("无文学编导专科录取规则!");
            }

            //判断考生是否有报取资格
            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() +
                    Double.parseDouble(schemeVo.getMajorGrade());
            double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() +
                    Double.parseDouble(schemeVo.getMajorGrade());
            if (secondCulture < secondRule.getCultureMin() || secondMajor < secondRule.getProfessionMin()) {
                return R.error("很抱歉，客户的专业成绩不满足报考条件，无法查询!");
            }
            list = getLiteratureVipList(schemeVo, firstRule, secondRule, total, local, nameByIds);
        }

        //(4)体育考生
        if (StudentType.SPORTS_STUDENT.getCode().equals(user.getExamType())) {
            //获取体育本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            if (null == firstRule) {
                return R.error("无体育本科录取规则!");
            }

            //获取体育专科一批录取规则
            AdmissionRule secondRule = admissionRuleMapper.getByType(SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode());
            if (null == secondRule) {
                return R.error("无体育专科录取规则!");
            }

            //判断考生是否有报取资格
            //综合成绩差值
            double value = secondRule.getCurrentYearCulture() - secondRule.getCultureMin();
            double culture = Double.parseDouble(schemeVo.getCultureGrade());
            double major = Double.parseDouble(schemeVo.getMajorGrade());

            //计算综合成绩
            double totalv = major * 750 / 100 * 0.7 + culture * 0.3 - value;
            if (totalv < secondRule.getCultureMin()) {
                return R.error("很抱歉，客户的成绩不满足报考条件，无法查询!");
            }

            list = getSportsVipList(totalv, firstRule, secondRule, total, local, nameByIds);
        }

        //4判断是否展示全部方案
        if (schemeVo.getIsShowAll() == 0) {
            List<Volunteer> result;
            if (list.size() < 10) {
                result = list.subList(0, list.size());
            } else {
                result = list.subList(0, 10);
            }
            //设置查询时间
            SetQueryTime(result);
            volunteerVo.setUserId(user.getId());
            volunteerVo.setVolunteers(result);
            return R.ok(volunteerVo);
        }

        //排序
        list.sort(Comparator.comparing(Volunteer::getLowestMark).reversed());
        volunteerVo.setUserId(user.getId());

        //设置查询时间
        SetQueryTime(list);
        volunteerVo.setVolunteers(list);
        volunteerVo.setTotal(list.size());
        return R.ok(volunteerVo);
    }

    /**
     * 获取vip文化生志愿列表
     *
     * @param firstRule       录取规则
     * @param professionNames 专业名称
     * @return vip文化生志愿列表
     */
    private List<Volunteer> getCultureVipList(AdmissionRule firstRule, AdmissionRule secondRule, Integer total, String local,
                                              List<String> professionNames,
                                              double firstCulture, double secondCulture) {
        double cultureMax = firstRule.getCultureMax();
        double cultureMin = firstRule.getCultureMin();

        //考生的分数在本科一批中
        List<Volunteer> list = new ArrayList<>();
        Map<String, Object> map = new HashMap<>(16);
        map.put("professionNames", professionNames);
        if (firstCulture < cultureMax && firstCulture >= cultureMin) {
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
            map.put("total", total);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);

        } else {
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());
            map.put("total", total);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
        }
        return list;
    }

    /**
     * 获取vip美术生志愿方案
     *
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getArtsVipList(SchemeVipVo schemeVo, AdmissionRule firstRule, AdmissionRule secondRule, Integer total, String local,
                                           List<String> professionNames) {
        //本科类比分 = 本年录取分 -去年录取分 + 考生分数
        double culture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(schemeVo.getCultureGrade());
        double major = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + Double.parseDouble(schemeVo.getMajorGrade());


        Map<String, Object> map = new HashMap<>(16);
        map.put("professionNames", professionNames);

        //1、考生是否能报取美术本科
        List<Volunteer> list = new ArrayList<>();
        if (culture >= firstRule.getCultureMin() && major >= firstRule.getProfessionMin()) {
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            map.put("total", total);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
            return list;
        }


        //专科科类比分 = 本年录取分 -去年录取分 + 考生分数
        double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() +
                Double.parseDouble(schemeVo.getCultureGrade());
        double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() +
                Double.parseDouble(schemeVo.getMajorGrade());

        //2、考生是否能报取美术专科
        if (secondCulture >= secondRule.getCultureMin() && secondMajor >= secondRule.getProfessionMin()) {

            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
            map.put("total", total);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
        }
        return list;
    }

    /**
     * 获取vip文学编导生志愿方案
     *
     * @param queryVo         考生成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getLiteratureVipList(SchemeVipVo queryVo, AdmissionRule firstRule, AdmissionRule secondRule, Integer total, String local,
                                                 List<String> professionNames) {
        //本科类比分 = 本年录取分 -去年录取分 + 考生分数
        double culture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
        double major = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());

        Map<String, Object> map = new HashMap<>(16);
        map.put("professionNames", professionNames);

        //1、考生是否能报取美术本科
        List<Volunteer> list = new ArrayList<>();
        if (culture >= firstRule.getCultureMin() && major >= firstRule.getProfessionMin()) {

            map.put("type", SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            map.put("total", total);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
            return list;
        }


        //专科类比分 = 本年录取分 -去年录取分 + 考生分数
        double cultureSecond = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
        double majorSecond = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());

        //2、考生是否能报取美术专科
        if (cultureSecond >= secondRule.getCultureMin() && majorSecond >= secondRule.getProfessionMin()) {
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            map.put("total", total);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
        }

        return list;
    }

    /**
     * 获取vip体育生生志愿方案
     *
     * @param total           考生综合成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getSportsVipList(double total, AdmissionRule firstRule, AdmissionRule secondRule, Integer totalv, String local,
                                             List<String> professionNames) {
        Map<String, Object> map = new HashMap<>(16);
        map.put("professionNames", professionNames);

        //1、考生是否能报取体育本科
        List<Volunteer> list = new ArrayList<>();
        if (total >= firstRule.getCultureMin()) {
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            map.put("total", totalv);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
            return list;
        }

        //2、考生是否能报取体育专科
        if (total >= secondRule.getCultureMin()) {
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode());
            map.put("total", totalv);
            map.put("local", local);
            list = volunteerMapper.getVolunteerListVip(map);
        }

        return list;
    }

    /**
     * 获取文化生志愿列表
     *
     * @param scoreInfo       成绩
     * @param firstRule       录取规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 文化生志愿列表
     */
    private List<Volunteer> getCultureList(ScoreInfo scoreInfo, AdmissionRule firstRule, AdmissionRule secondRule,
                                           List<String> classNames, List<String> professionNames,
                                           double firstCulture, double secondCulture) {
        double cultureMax = firstRule.getCultureMax();
        double cultureMin = firstRule.getCultureMin();

        //考生的分数在本科一批中
        List<Volunteer> list = new ArrayList<>();
        Map<String, Object> map = new HashMap<>(16);
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);
        if (firstCulture < cultureMax && firstCulture >= cultureMin) {
            double upGrade = firstCulture + firstRule.getUpMark();
            double downGrade = firstCulture - firstRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
            map.put("downGrade", downGrade);
            list = volunteerMapper.getVolunteerList(map);
            if (list.size() < firstRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, firstRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
                list.addAll(fillList);
            }
        } else {
            //考生的分数是在专科一批中
            double upGrade = secondCulture + secondRule.getUpMark();
            double downGrade = secondCulture - secondRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());
            map.put("downGrade", downGrade);
            list = volunteerMapper.getVolunteerList(map);

            //如果数据不满足最大数量，取不限专业最低分补充
            if (list.size() < secondRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, secondRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode());
                list.addAll(fillList);
            }
        }
        return list;
    }

    /**
     * 获取美术生志愿方案
     *
     * @param queryVo         考生成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getArtsList(SchemeVo queryVo, AdmissionRule firstRule, AdmissionRule secondRule,
                                        List<String> classNames, List<String> professionNames) {
        //本科类比分 = 本年录取分 -去年录取分 + 考生分数
        double culture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
        double major = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());
        //计算综合成绩
        double total = major * 750 / 300 * 0.7 + culture * 0.3;


        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);

        //1、考生是否能报取美术本科
        List<Volunteer> list = new ArrayList<>();
        if (culture >= firstRule.getCultureMin() && major >= firstRule.getProfessionMin()) {
            //浮动分数
            double upGrade = total + firstRule.getUpMark();
            double downGrade = total - firstRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            list = volunteerMapper.getVolunteerList(map);

            //如果数据不满足最大数量，取不限专业最低分补充
            if (list.size() < secondRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, firstRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
                list.addAll(fillList);
            }
            return list;
        }


        //专科科类比分 = 本年录取分 -去年录取分 + 考生分数
        double secondCulture = secondRule.getCurrentYearCulture() - secondRule.getCultureMin() +
                Double.parseDouble(queryVo.getCultureGrade());
        double secondMajor = secondRule.getCurrentYearMajor() - secondRule.getProfessionMin() +
                Double.parseDouble(queryVo.getMajorGrade());

        //计算综合成绩
        double secondRuleTotal = secondMajor * 750 / 300 * 0.7 + secondCulture * 0.3;

        //判断文化成绩大于本科文化成绩最低分但专业成绩小于本科文化成绩最低分时进行文化成绩降分处理查询
        if (secondCulture > firstRule.getCultureMin() && secondMajor < firstRule.getProfessionMin()) {
            List<Volunteer> volunteers = new ArrayList<>();
            volunteers = volunteerMapper.getVolunteerByType(SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            secondRuleTotal = Double.valueOf(volunteers.get(0).getLowestMark()) - secondRule.getUpMark() + 1;
        }

        //2、考生是否能报取美术专科
        if (secondCulture >= secondRule.getCultureMin() && secondMajor >= secondRule.getProfessionMin()) {

            //浮动分数
            double upGrade = secondRuleTotal + secondRule.getUpMark();
            double downGrade = secondRuleTotal - secondRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
            list = volunteerMapper.getVolunteerList(map);

            //如果数据不满足最大数量，取不限专业最低分补充
            if (list.size() < secondRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, secondRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode());
                list.addAll(fillList);
            }
        }
        return list;
    }

    /**
     * 获取文学编导生志愿方案
     *
     * @param queryVo         考生成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getLiteratureList(SchemeVo queryVo, AdmissionRule firstRule, AdmissionRule secondRule,
                                              List<String> classNames, List<String> professionNames) {
        //本科类比分 = 本年录取分 -去年录取分 + 考生分数
        double culture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
        double major = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());
        //计算综合成绩
        double total = major * 750 / 300 * 0.3 + culture * 0.7;

        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);

        //1、考生是否能报取美术本科
        List<Volunteer> list = new ArrayList<>();
        if (culture >= firstRule.getCultureMin() && major >= firstRule.getProfessionMin()) {
            //浮动分数
            double upGrade = total + firstRule.getUpMark();
            double downGrade = total - firstRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            list = volunteerMapper.getVolunteerList(map);

            //如果数据不满足最大数量，取不限专业最低分补充
            if (list.size() < secondRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, secondRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
                list.addAll(fillList);
            }
            return list;
        }


        //专科类比分 = 本年录取分 -去年录取分 + 考生分数
        double cultureSecond = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + Double.parseDouble(queryVo.getCultureGrade());
        double majorSecond = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + Double.parseDouble(queryVo.getMajorGrade());

        //计算综合成绩
        double totalSecond = majorSecond * 750 / 300 * 0.3 + cultureSecond * 0.7;

        //判断文化成绩大于本科文化成绩最低分但专业成绩小于本科文化成绩最低分时进行文化成绩降分处理查询
        if (cultureSecond > firstRule.getCultureMin() && majorSecond < firstRule.getProfessionMin()) {
            List<Volunteer> volunteers = new ArrayList<>();
            volunteers = volunteerMapper.getVolunteerByType(SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            totalSecond = Double.valueOf(volunteers.get(0).getLowestMark()) - secondRule.getUpMark() + 1;
        }

        log.info(classNames.toString());
        //2、考生是否能报取美术专科
        if (cultureSecond >= secondRule.getCultureMin() && majorSecond >= secondRule.getProfessionMin()) {
            //浮动分数
            double upGrade = totalSecond + secondRule.getUpMark();
            double downGrade = totalSecond - secondRule.getDownMark();
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
            list = volunteerMapper.getVolunteerList(map);

            log.info(Integer.toString(list.size()));
            //如果数据不满足最大数量，取不限专业最低分补充
            if (list.size() < secondRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, secondRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode());
                list.addAll(fillList);
            }
        }

        return list;
    }

    /**
     * 获取体育生生志愿方案
     *
     * @param total           考生综合成绩
     * @param firstRule       本科规则
     * @param secondRule      专科规则
     * @param classNames      副科名称
     * @param professionNames 专业名称
     * @return 志愿方案列表
     */
    private List<Volunteer> getSportsList(double total, AdmissionRule firstRule, AdmissionRule secondRule,
                                          List<String> classNames, List<String> professionNames) {
        Map<String, Object> map = new HashMap<>(16);
        map.put("number", firstRule.getNumber());
        map.put("classNames", classNames);
        map.put("professionNames", professionNames);

        //1、考生是否能报取体育本科
        List<Volunteer> list = new ArrayList<>();
        if (total >= firstRule.getCultureMin()) {
            //浮动分数
            double upGrade = total + firstRule.getUpMark();
            double downGrade = total - firstRule.getDownMark();
            if (downGrade < firstRule.getCultureMin()) {
                downGrade = firstRule.getCultureMin();
            }
            map.put("upGrade", upGrade);
            map.put("downGrade", downGrade);
            map.put("type", SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            list = volunteerMapper.getVolunteerList(map);

            //如果数据不满足最大数量，取不限专业最低分补充
            if (list.size() < secondRule.getNumber()) {
                List<Volunteer> fillList = fillUpVolunteer(list, secondRule.getNumber(), upGrade, downGrade,
                        SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
                list.addAll(fillList);
            }
            return list;
        }

        //2、考生是否能报取体育专科
        if (total >= secondRule.getCultureMin()) {
            list = volunteerMapper.getVolunteerByType(SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode());
        }

        return list;
    }

    /**
     * 获取副科的名称
     *
     * @param scoreInfo 成绩信息
     * @return 副科列表
     */
    private List<String> getClassName(ScoreInfo scoreInfo) {
        List<String> name = new ArrayList<>();
        name.add(StudentClassEnum.UNLIMITED.getDesc());
        if (null != scoreInfo.getPhysicsGrade() && scoreInfo.getPhysicsGrade() >= 0) {
            name.add(StudentClassEnum.PHYSICS_GRADE.getDesc());
        }

        if (null != scoreInfo.getChemistryGrade() && scoreInfo.getChemistryGrade() >= 0) {
            name.add(StudentClassEnum.CHEMISTRY_GRADE.getDesc());
        }

        if (null != scoreInfo.getBiologyGrade() && scoreInfo.getBiologyGrade() >= 0) {
            name.add(StudentClassEnum.BIOLOGY_GRADE.getDesc());
        }

        if (null != scoreInfo.getHistoryGrade() && scoreInfo.getHistoryGrade() >= 0) {
            name.add(StudentClassEnum.HISTORY_GRADE.getDesc());
        }

        if (null != scoreInfo.getGeographyGrade() && scoreInfo.getGeographyGrade() >= 0) {
            name.add(StudentClassEnum.GEOGRAPHY_GRADE.getDesc());
        }

        if (null != scoreInfo.getPoliticsGrade() && scoreInfo.getPoliticsGrade() >= 0) {
            name.add(StudentClassEnum.POLITICS_GRADE.getDesc());
        }

        return name;
    }

    /**
     * 获取专业名称
     *
     * @param schemeVo 专业参数
     * @return 专业名称列表
     */
    private List<String> getProfessionName(SchemeVo schemeVo) {
        //把字符专ids业转为数组
        List<Integer> professionIds = StringUtil.strToArray(schemeVo.getProfessionIds());
        return professionMapper.getNameByIds(professionIds);
    }


    /**
     * 补充不足数量的志愿数
     *
     * @param list   志愿ids
     * @param number 差的数量
     * @param max    最高分
     * @param min    最低分
     * @return 集合
     */
    private List<Volunteer> fillUpVolunteer(List<Volunteer> list, Integer number, double max, double min, Integer type) {

        //已存在的志愿ids
        List<Integer> volunteerIds = list.stream().map(Volunteer::getId).collect(Collectors.toList());

        //需要补充的志愿条数
        int totalNumber = number - list.size();

        Map<String, Object> map = new HashMap<>(16);
        map.put("volunteerIds", volunteerIds);
        map.put("upGrade", max);
        map.put("downGrade", min);
        map.put("type", type);

        List<Volunteer> fillList = volunteerMapper.getFillList(map);
        List<Volunteer> resultList = new ArrayList<>();
        List<Volunteer> otherList = new ArrayList<>();
        Map<Double, List<Volunteer>> mapObj = fillList.stream().collect(Collectors.groupingBy(Volunteer::getLowestMark
        ));

        for (Map.Entry<Double, List<Volunteer>> entry : mapObj.entrySet()) {
            List<Volunteer> volunteers = entry.getValue();
            if (volunteers.size() > 1) {
                Random mRandom = new Random();
                int count = mRandom.nextInt(volunteers.size());
                resultList.add(volunteers.get(count));
                volunteers.remove(count);
                otherList.addAll(volunteers);
            } else {
                resultList.addAll(volunteers);
            }

            if (resultList.size() >= totalNumber) {
                return resultList.subList(0, totalNumber);
            }
        }

        //从剩下的数据中随机抽取，添满数组
        if (resultList.size() < totalNumber) {
            int num = totalNumber - resultList.size();
            List<Volunteer> volunteers = new ArrayList<>();
            if (otherList.size() > 0) {
                for (int i = 0; i < num; i++) {
                    Random mRandom = new Random();
                    if (otherList.size() > 0) {
                        int count = mRandom.nextInt(otherList.size());
                        if (resultList.contains(otherList.get(count))) {
                            otherList.remove(count);
                            continue;
                        }
                        volunteers.add(otherList.get(count));
                        otherList.remove(count);
                    } else {
                        break;
                    }
                }
            }
            resultList.addAll(volunteers);
        }
        return resultList;
    }

    /**
     * 分页处理方法
     *
     * @param schemeVo 参数
     */
    private void pageHandle(SchemeQueryVo schemeVo) {
        Integer pageNum = schemeVo.getPageNo();
        Integer pageSize = schemeVo.getPageSize();
        if (null == pageSize || pageSize == 0) {
            pageSize = 10;
        }
        if (null == pageNum || pageNum == 0) {
            pageNum = 1;
        }
        Integer endNum = pageSize;
        Integer startNum = (pageNum - CommonAttributes.NUM_ONE) * pageSize;
        schemeVo.setEndNum(endNum);
        schemeVo.setStartNum(startNum);
        schemeVo.setPageNo(pageNum);
        schemeVo.setPageSize(pageSize);
    }


    /**
     * 超过限定展示方案条数，要帅选
     *
     * @param list  所以方案数据
     * @param total 限定展示方案条数
     * @return 筛选结果
     */
    private List<Volunteer> getVolunteerList(List<Volunteer> list, Integer total) {
        List<Volunteer> result = new ArrayList<>();
        List<Volunteer> otherList = new ArrayList<>();

        //分组，每组数量大于2的，取2条，小于等于则全取
        Map<Double, List<Volunteer>> map = list.stream().collect(Collectors.groupingBy(Volunteer::getLowestMark
        ));

        for (Map.Entry<Double, List<Volunteer>> entry : map.entrySet()) {
            List<Volunteer> volunteers = entry.getValue();
            if (volunteers.size() > 2) {
                for (int i = 0; i < 2; i++) {
                    Random mRandom = new Random();
                    int number = mRandom.nextInt(volunteers.size());
                    result.add(volunteers.get(number));
                    volunteers.remove(number);
                }
                otherList.addAll(volunteers);
            } else {
                result.addAll(volunteers);
            }

            if (result.size() >= total) {
                return result.subList(0, total);
            }
        }

        //分组，每组数量大于2的，取2条，小于等于则全取
        Map<Double, List<Volunteer>> mapOther = otherList.stream().collect(Collectors.groupingBy(Volunteer::getLowestMark
        ));
        List<Volunteer> endList = new ArrayList<>();
        for (Map.Entry<Double, List<Volunteer>> entry : mapOther.entrySet()) {
            List<Volunteer> volunteers = entry.getValue();
            if (volunteers.size() > 2) {
                for (int i = 0; i < 2; i++) {
                    Random mRandom = new Random();
                    int number = mRandom.nextInt(volunteers.size());
                    result.add(volunteers.get(number));
                    volunteers.remove(number);
                }
                endList.addAll(volunteers);
            } else {
                result.addAll(volunteers);
            }

            if (result.size() >= total) {
                return result.subList(0, total);
            }
        }


        //从剩下的数据中随机抽取，添满数组
        if (result.size() < total) {
            int number = total - result.size();
            List<Volunteer> volunteers = new ArrayList<>();
            for (int i = 0; i < number; i++) {
                Random mRandom = new Random();
                int count = mRandom.nextInt(endList.size());
                if (result.contains(endList.get(count))) {
                    endList.remove(count);
                    continue;
                }
                volunteers.add(endList.get(count));
                endList.remove(count);
            }
            result.addAll(volunteers);
        }
        return result;
    }

    /**
     * 设置查询时间
     *
     * @param list 志愿列表
     */
    private void SetQueryTime(List<Volunteer> list) {
        if (!CollectionUtils.isEmpty(list)) {
            for (Volunteer volunteer : list) {
                volunteer.setQueryTime(new Date());
            }
        }
    }

}
