package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.constant.CommonAttributes;
import cn.wisenergy.common.utils.R;
import cn.wisenergy.mapper.*;
import cn.wisenergy.model.app.*;
import cn.wisenergy.model.dto.UserCommitDto;
import cn.wisenergy.model.dto.UserDto;
import cn.wisenergy.model.dto.UserInfoDto;
import cn.wisenergy.model.vo.UserInfoVo;
import cn.wisenergy.model.vo.UserQueryVo;
import cn.wisenergy.model.vo.UserShowVo;
import cn.wisenergy.model.vo.UserVipCommitVo;
import cn.wisenergy.service.app.UserService;
import cn.wisenergy.service.common.VolunteerManager;
import cn.wisenergy.service.util.WxUtil;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageInfo;
import io.swagger.models.auth.In;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletResponse;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @ Description: 用户接口实现
 * @ Author     : 86187
 * @ Date       : 2021/1/6 16:11
 */
@Service
@Slf4j
public class UserServiceImpl extends ServiceImpl<UsersMapper, User> implements UserService {

    @Autowired
    private UsersMapper usersMapper;

    @Autowired
    private ScoreInfoMapper scoreInfoMapper;

    @Autowired
    private LoginRecordMapper loginRecordMapper;

    @Autowired
    private UserLimitMapper userLimitMapper;

    @Autowired
    private PayRecordMapper payRecordMapper;

    @Autowired
    protected VolunteerManager volunteerManager;

    @Value("${wxPay.Applets_ID}")
    private String appid;

    @Value("${wxPay.SECRT_KEY}")
    private String key;

    private static final int ERROR_CODE = 1;

    @Override
    public R<PageInfo<UserInfoDto>> getUserList(UserQueryVo queryVo) {
        log.info("volunteer_service[]UserServiceImpl[]getUserList[]input.param.queryVo:" + queryVo);

        //创建参数容器
        Map<String, Object> map = new HashMap<>(4);

        //将参数放入容器中

        if (null!=queryVo.getPageNo()||null!=queryVo.getPageSize()){
            map.put("pageNo", (queryVo.getPageNo() - 1) * queryVo.getPageSize());
            map.put("pageSize", queryVo.getPageSize());
        }
        map.put("startTime", queryVo.getStartTime());
        map.put("endTime", queryVo.getEndTime());
        map.put("userName", queryVo.getUserName());
        map.put("phone", queryVo.getPhone());

        //查询用户数据
        List<User> list1 = usersMapper.getList(map);
        List<UserInfoDto> list = new ArrayList<>();

        //将集合遍历拿出用户id查询成绩信息
        for (User user : list1) {
            UserInfoDto userInfoDto = new UserInfoDto();

            //查询相关信息
            ScoreInfo scoreInfo = scoreInfoMapper.getByUserId(user.getId());
            if (null != scoreInfo) {
                userInfoDto.setCultureGrade(scoreInfo.getCultureGrade().toString());
                userInfoDto.setMajorGrade(null==scoreInfo.getMajorGrade()?"":scoreInfo.getMajorGrade().toString());
            }
                LoginRecord loginRecord = loginRecordMapper.getById(user.getId());
                if (null != loginRecord) {
                    userInfoDto.setLastLoginTime(loginRecord.getUpdateTime());
                    userInfoDto.setIp(loginRecord.getIp());
                }
                UserLimit userLimit = userLimitMapper.getByUserId(user.getId());

                if (null != userLimit) {
                    userInfoDto.setQueryLimit(userLimit.getUsableLimit());
                }
                List<PayRecord> byId = payRecordMapper.getById(user.getId());

            //计算充值卡充值次数
            int rechargeTimes = 0;

            //支付宝充值总金额
            int alipayMoney = 0;

            //微信充值总金额
            int weChatMoney = 0;

            if (!CollectionUtils.isEmpty(byId)) {

                for (PayRecord payRecord : byId) {
                    if (payRecord.getType() == 1) {
                        alipayMoney += null == payRecord.getMoney() ? 0 : payRecord.getMoney();
                    }

                    if (payRecord.getType() == 2) {
                        weChatMoney += null == payRecord.getMoney() ? 0 : payRecord.getMoney();
                    }

                    if (payRecord.getType() == 3) {
                        rechargeTimes += 1;
                    }
                }
            }
            userInfoDto.setWeChatMoney(String.valueOf(weChatMoney));
            userInfoDto.setAlipayMoney(String.valueOf(alipayMoney));
            userInfoDto.setRechargeTimes(rechargeTimes);

                //将消息进行同步
                BeanUtils.copyProperties(user, userInfoDto);
                userInfoDto.setUserId(user.getId());
                userInfoDto.setRegisterTime(user.getCreateTime());
                list.add(userInfoDto);
            }

            PageInfo<UserInfoDto> info = new PageInfo<>();
            info.setPageSize(null==queryVo.getPageSize()?0:queryVo.getPageSize());
            info.setPageNum(null==queryVo.getPageNo()?0:queryVo.getPageNo());
            info.setTotal(usersMapper.getUserNumbers(map));
            info.setList(list);

            return R.ok(info);
        }


    @Override
    public R<UserInfoVo> commitUserInfo(UserCommitDto userInfo) {
        log.info("volunteer-service[]UserServiceImpl[]commitUserInfo[]input.param.userInfo:" + userInfo);
        if (null == userInfo) {
            return R.error("入参为空!");
        }

        //1、根据用户id,获取用户信息
        User user = usersMapper.selectById(userInfo.getUserId());
        if (null == user) {
            return R.error("用户信息不存在!");
        }

        //2、要提交的用户信息
        user.setSchool(userInfo.getSchool());
        user.setSex(userInfo.getSex());
        user.setExamType(userInfo.getStudentType());
        user.setUserName(userInfo.getUserName());
        user.setIsDelete(0);
        user.setIsVip(2);
        user.setSource(userInfo.getSource());

        //3、要提交的用户的成绩信息
        ScoreInfo scoreInfo = userInfo.getScoreInfo();

        //检查副科
        R<Boolean> checkScore = volunteerManager.checkScore(scoreInfo);
        if (null != checkScore && checkScore.getCode() == ERROR_CODE) {
            return R.error(checkScore.getMessage());
        }

        ScoreInfo flagStatus = scoreInfoMapper.getByUserId(user.getId());
        int flag;
        if (null == flagStatus) {
            //新增
            flag = 1;
        } else {
            //编辑
            flag = 2;
        }

        //4、更新用户信息和保存用户成绩
        boolean bool = volunteerManager.updateAndSaveUserInfo(user, scoreInfo,flag);
        if (!bool) {
            return R.error("添加用户信息失败!");
        }

        //返回信息
        UserInfoVo userInfoVo = new UserInfoVo();
        userInfoVo.setUserId(userInfo.getUserId());
        userInfoVo.setUserName(userInfo.getUserName());
        userInfoVo.setPhone(user.getPhone());
        userInfoVo.setSex(userInfo.getSex());
        userInfoVo.setStudentType(userInfo.getStudentType());
        userInfoVo.setUserName(userInfo.getUserName());
        return R.ok(userInfoVo);
    }

    @Override
    public R<Boolean> edit(UserCommitDto userDto) {
        if (null == userDto || StringUtils.isBlank(userDto.getUserName())) {
            return R.error("入参为空!");
        }

        //1、根据用户id,获取用户信息
        User user = usersMapper.selectById(userDto.getUserId());
        if (null == user) {
            return R.error("用户信息不存在!");
        }

        //2、要提交的用户信息
        user.setSchool(userDto.getSchool());
        user.setSex(userDto.getSex());
        user.setExamType(userDto.getStudentType());
        user.setUserName(userDto.getUserName());


        //3、要提交的用户的成绩信息
        ScoreInfo scoreInfo = scoreInfoMapper.getByUserId(userDto.getUserId());

        //检查副科
        R<Boolean> checkScore = volunteerManager.checkScore(userDto.getScoreInfo());
        if (null != checkScore && checkScore.getCode() == ERROR_CODE) {
            return R.error(checkScore.getMessage());
        }

        scoreInfo.setUserId(userDto.getUserId());
        scoreInfo.setBiologyGrade(userDto.getScoreInfo().getBiologyGrade());
        scoreInfo.setChemistryGrade(userDto.getScoreInfo().getChemistryGrade());
        scoreInfo.setCultureGrade(userDto.getScoreInfo().getCultureGrade());
        scoreInfo.setEnglishGrade(userDto.getScoreInfo().getEnglishGrade());
        scoreInfo.setGeographyGrade(userDto.getScoreInfo().getGeographyGrade());
        scoreInfo.setHistoryGrade(userDto.getScoreInfo().getHistoryGrade());
        scoreInfo.setLanguageGrade(userDto.getScoreInfo().getLanguageGrade());
        scoreInfo.setMajorGrade(userDto.getScoreInfo().getMajorGrade());
        scoreInfo.setMathGrade(userDto.getScoreInfo().getMathGrade());
        scoreInfo.setPhysicsGrade(userDto.getScoreInfo().getPhysicsGrade());
        scoreInfo.setPoliticsGrade(userDto.getScoreInfo().getPoliticsGrade());

        //4、更新用户信息和保存用户成绩
        boolean bool = volunteerManager.updateAndUserInfo(user, scoreInfo);

        return R.ok(bool);
    }

    @Override
    public R<Boolean> delete(Integer userId) {
        return null;
    }

    @Override
    public R<UserShowVo> getById(Integer id) {
        log.info("volunteer-service[]UserServiceImpl[]getById[]input.param.id:" + id);
        if (null == id) {
            return R.error("入参为空!");
        }

        //根据id,获取用户信息
        User user = usersMapper.selectById(id);
        if (null == user) {
            return R.error("用户信息不存在!");
        }

        UserShowVo userShowVo = new UserShowVo();
        userShowVo.setId(user.getId());
        userShowVo.setSchool(user.getSchool());
        userShowVo.setSex(user.getSex());
        userShowVo.setStudentType(user.getExamType());
        userShowVo.setUserName(user.getUserName());

        //获取成绩信息
        QueryWrapper<ScoreInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", user.getId());
        ScoreInfo scoreInfo = scoreInfoMapper.selectOne(queryWrapper);
        if (null != scoreInfo) {
            userShowVo.setScoreInfo(scoreInfo);
        }
        return R.ok(userShowVo);
    }

    @Override
    public User getByPhone(String phone) {
        return null;
    }

    @Override
    public R<Boolean> createExcel(UserQueryVo queryVo, HttpServletResponse response) {
        log.info("UserServiceImpl[].createExcel[].input.param:"+queryVo);

        queryVo.setPageSize(null);
        queryVo.setPageNo(null);
        //生成Excel
        try {
            //  设置内容格式 以及 编码方式
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");

            List<UserInfoDto> list=getUserList(queryVo).getData().getList();

            //  使用java8新特性的stream流去处理数据，把空的数据过滤掉
            List<UserDto> resultBo = list.stream().filter(Objects::nonNull)
                    .map(t -> UserDto.builder()
                            .registerTime(t.getRegisterTime())
                            .source(t.getSource()==1?"PC":"APP")
                            .userName(t.getUserName())
                            .phone(t.getPhone())
                            .lastLoginTime(t.getLastLoginTime())
                            .school(t.getSchool())
                            .examType(t.getExamType()==1?"文化生":t.getExamType()==2?"美术生":t.getExamType()==3?"体育生":"文学编导考生")
                            .sex(t.getSex()==0?"男":"女")
                            .rechargeTimes(t.getRechargeTimes())
                            .weChatMoney(t.getWeChatMoney())
                            .alipayMoney(t.getAlipayMoney())
                            .lastLoginTime(t.getLastLoginTime())
                            .cultureGrade(t.getCultureGrade())
                            .majorGrade(t.getMajorGrade())
                            .queryLimit(t.getQueryLimit())
                            .build()).collect(Collectors.toList());

            //创建文件名称
            long lon = System.currentTimeMillis();

            response.setHeader("Content-disposition", "attachment;filename=" + lon + ".xlsx");

            //  sheet名称
            EasyExcel.write(response.getOutputStream(), UserDto.class).sheet(Long.toString(lon)).doWrite(resultBo);
            return R.ok(0,true);
        } catch (Exception e) {
            return R.ok(1,false);
        }
    }

    @Override
    public R<List<User>> test() {
        List<String> names=new ArrayList<>();
        names.add("黎楚川");
        names.add("陈成");
        names.add("雷青松");
        List<User> list=usersMapper.test(names);
        return R.ok(list);
    }

    /**
     * 分页处理方法
     *
     * @param orderQueryVo 参数
     */
    private void pageHandle(UserQueryVo orderQueryVo) {
        Integer pageNum = orderQueryVo.getPageNo();
        Integer pageSize = orderQueryVo.getPageSize();
        if (null == pageSize || pageSize == 0) {
            pageSize = 10;
        }
        if (null == pageNum || pageNum == 0) {
            pageNum = 1;
        }
        Integer endNum = pageSize;
        Integer startNum = (pageNum - CommonAttributes.NUM_ONE) * pageSize;
        orderQueryVo.setEndNum(endNum);
        orderQueryVo.setStartNum(startNum);
        orderQueryVo.setPageNo(pageNum);
        orderQueryVo.setPageSize(pageSize);
    }

    @Override
    public R<UserInfoVo> commitUserVipInfo(UserVipCommitVo userInfo) {
        log.info("volunteer-service[]UserServiceImpl[]commitUserVipInfo[]input.param.userInfo:" + userInfo);
        if (null == userInfo||null==userInfo.getStaffId()|| null==userInfo.getSource()) {
            return R.error("入参为空!");
        }

        User user = new User();
        //2、要提交的用户信息
        user.setSchool(userInfo.getSchool());
        user.setSex(userInfo.getSex());
        user.setExamType(userInfo.getStudentType());
        user.setUserName(userInfo.getUserName());
        user.setIsDelete(0);
        user.setIsVip(1);
        user.setId(userInfo.getUserId());
        user.setSource(userInfo.getSource());

        //3、要提交的用户的成绩信息
        ScoreInfo scoreInfo = userInfo.getScoreInfo();

        //检查副科
        R<Boolean> checkScore = volunteerManager.checkScore(scoreInfo);
        if (null != checkScore && checkScore.getCode() == ERROR_CODE) {
            return R.error(checkScore.getMessage());
        }

        //设置flag为添加用户
        userInfo.setFlag(1);

        //4、更新用户信息和保存用户成绩
        boolean bool = volunteerManager.updateAndSaveUserVipInfo(user, scoreInfo,userInfo.getFlag(),userInfo.getStaffId());
        if (!bool) {
            return R.error("添加用户信息失败!");
        }


        //返回信息
        UserInfoVo userInfoVo = new UserInfoVo();
        userInfoVo.setUserId(userInfo.getUserId());
        userInfoVo.setUserName(userInfo.getUserName());
        userInfoVo.setPhone(user.getPhone());
        userInfoVo.setSex(userInfo.getSex());
        userInfoVo.setStudentType(userInfo.getStudentType());
        userInfoVo.setUserName(userInfo.getUserName());
        return R.ok(userInfoVo);
    }

    @Override
    public R<String> appletBinding(String code,Integer userId) {
        if (StringUtils.isBlank(code)||userId==null) {
            return R.error("入参为空!");
        }
        Map<String, Object> map = WxUtil.getInfoByCode(code, appid, key);
        log.info("code解析数据："+map);
        if(null == map.get("openid")){
            return R.error(map.get("errmsg").toString());
        }
        User user = new User();
        user.setOpenid(map.get("openid").toString());
        user.setId(userId);
        user.setUpdateTime(new Date());
        int update = baseMapper.updateById(user);
        if(update>0){
            return R.ok("授权成功");
        }
        return R.error("授权失败");
    }

    @Override
    public R<UserInfoVo> getByOpenid(String code) {
        if (StringUtils.isBlank(code)) {
            return R.error("入参为空!");
        }
        Map<String, Object> map = WxUtil.getInfoByCode(code, appid, key);
        log.info("code解析数据："+map);

        if(null == map.get("openid")){
            return R.error(map.get("errmsg").toString());
        }
        QueryWrapper<User> objectQueryWrapper = new QueryWrapper<>();
        objectQueryWrapper.eq("openid",map.get("openid").toString());
        User user = baseMapper.selectOne(objectQueryWrapper);
        if(null == user){
            return R.error("该用户还没有授权");
        }
        UserInfoVo userInfoVo = new UserInfoVo();
        userInfoVo.setUserId(user.getId());
        userInfoVo.setUserName(user.getUserName());
        userInfoVo.setPhone(user.getPhone());
        userInfoVo.setSex(user.getSex());
        userInfoVo.setStudentType(user.getExamType());
        return R.ok(userInfoVo);
    }

    @Override
    public R<String> quitApplet(Integer userId) {
        if (userId == null) {
            return R.error("入参为空!");
        }
        User user = new User();
        user.setId(userId);
        user.setOpenid("");
        user.setUpdateTime(new Date());
        int update = baseMapper.updateById(user);
        if(update>0){
            return R.ok("退出成功");
        }
        return R.error("退出失败");
    }
}

