package cn.wisenergy.web.shir.config;

import cn.wisenergy.web.shir.cache.MySessionManager;
import cn.wisenergy.web.shir.filter.AuthenticationFilter;
import cn.wisenergy.web.shir.filter.KickoutSessionControlFilter;
import cn.wisenergy.web.shir.realm.AdminRealm;
import cn.wisenergy.web.shir.realm.CodeRealm;
import cn.wisenergy.web.shir.realm.StaffRealm;
import cn.wisenergy.web.shir.realm.UserRealm;
import cn.wisenergy.web.shir.util.UserModularRealmAuthenticator;
import org.apache.shiro.authc.pam.AtLeastOneSuccessfulStrategy;
import org.apache.shiro.authc.pam.ModularRealmAuthenticator;
import org.apache.shiro.realm.Realm;
import org.apache.shiro.spring.security.interceptor.AuthorizationAttributeSourceAdvisor;
import org.apache.shiro.spring.web.ShiroFilterFactoryBean;
import org.apache.shiro.web.mgt.DefaultWebSecurityManager;
import org.crazycake.shiro.IRedisManager;
import org.crazycake.shiro.RedisCacheManager;
import org.crazycake.shiro.RedisManager;
import org.crazycake.shiro.RedisSessionDAO;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@Configuration
public class ShiroConfig {

    //redis地址
    @Value("${spring.redis.host}")
    private String host;
    //redis端口
    @Value("${spring.redis.port}")
    private int port;
    //redis连接超时时间
    @Value("${spring.redis.timeout}")
    private String timeout;
    //redis密码
    @Value("${spring.redis.password}")
    private String password;
    //设置session会话过期时间为两小时
    private static final Integer expireTime = 3600 * 2;


    /**
     * 创建ShrioFilterFactoryBean
     */
    @Bean("shiroFilter")
    public ShiroFilterFactoryBean getShiroFilterFactoryBean(DefaultWebSecurityManager securityManager){
        ShiroFilterFactoryBean shiroFilterFactoryBean = new ShiroFilterFactoryBean();
        shiroFilterFactoryBean.setSecurityManager(securityManager);

        //添加shrio内置过滤器
        /**
         * 常用的过滤器：
         * anon:无需认证（登录）就可以访问
         * authc：必须认证才能访问
         * user：使用rememberMe功能可以直接访问
         * perms:该资源必须得到资源权限才能访问
         * role:该资源必须得到角色权限才能访问
         */
        Map<String,String> filterChainDefinitionMap=new LinkedHashMap<>();

        //自定义拦截器
        Map<String, Filter> filtersMap = new LinkedHashMap<>();
        filtersMap.put("authc", new AuthenticationFilter());
        filtersMap.put("kickout",kickoutSessionControlFilter());
        shiroFilterFactoryBean.setFilters(filtersMap);

        filterChainDefinitionMap.put("/authInformation/save", "anon");//存储设备IMEI号和手机SIM卡ID号
        filterChainDefinitionMap.put("/user/login/*", "anon"); // 登录页面-身份认证
        filterChainDefinitionMap.put("/user/sendSms", "anon"); // 发送验证码
        filterChainDefinitionMap.put("/user/valid", "anon"); // 缓存验证码
        filterChainDefinitionMap.put("/account/login", "anon"); // 登录页面-身份认证
        filterChainDefinitionMap.put("/staffUser/login", "anon"); // 登录页面-身份认证
        filterChainDefinitionMap.put("/swagger-ui.html", "anon"); // swagger接口-匿名访问
        filterChainDefinitionMap.put("/swagger/**", "anon");
        filterChainDefinitionMap.put("/admin/anon/**", "anon");
        filterChainDefinitionMap.put("/webjars/springfox-swagger-ui/**", "anon");
        filterChainDefinitionMap.put("/swagger-resources/**", "anon");
        filterChainDefinitionMap.put("/v2/api-docs", "anon");
        filterChainDefinitionMap.put("/upload_flowChart/**", "anon");//图片地址
        filterChainDefinitionMap.put("/webSocket/**", "anon");//socket
        filterChainDefinitionMap.put("/message/**", "anon");//消息推送接口
        filterChainDefinitionMap.put("/**", "authc");
        filterChainDefinitionMap.put("/account/**", "kickout");
        filterChainDefinitionMap.put("/banner/**", "kickout");
        filterChainDefinitionMap.put("/pic/**", "kickout");
        filterChainDefinitionMap.put("/school/**", "kickout");
        filterChainDefinitionMap.put("/pay/**", "kickout");
        filterChainDefinitionMap.put("/price/**", "kickout");
        filterChainDefinitionMap.put("/profession/**", "kickout");
        filterChainDefinitionMap.put("/refillCard/**", "kickout");
        filterChainDefinitionMap.put("/scheme/**", "kickout");
        filterChainDefinitionMap.put("/user/**", "kickout");
        filterChainDefinitionMap.put("/staff/**", "kickout");
        filterChainDefinitionMap.put("/staffUser/**", "kickout");
        filterChainDefinitionMap.put("/volunteer/**", "kickout");
        shiroFilterFactoryBean.setFilterChainDefinitionMap(filterChainDefinitionMap);
        return shiroFilterFactoryBean;
    }

    /**
     * 创建DefaultWebSecurityManager
     */
    @Bean
    public DefaultWebSecurityManager securityManager() {
        DefaultWebSecurityManager securityManager = new DefaultWebSecurityManager();
        securityManager.setAuthenticator(modularRealmAuthenticator());
        List<Realm> realms = new ArrayList<>();
        realms.add(adminRealm());
        realms.add(userRealm());
        realms.add(staffRealm());
        realms.add(codeRealm());
        securityManager.setRealms(realms);
        // 自定义session管理 使用redis
        securityManager.setSessionManager(sessionManager());
        // 自定义缓存实现 使用redis
        securityManager.setCacheManager(cacheManagers());
        return securityManager;
    }

    /**
     * 创建Realm
     */
    @Bean
    public ModularRealmAuthenticator modularRealmAuthenticator(){
        //自己重写的ModularRealmAuthenticator
        UserModularRealmAuthenticator modularRealmAuthenticator = new UserModularRealmAuthenticator();
        modularRealmAuthenticator.setAuthenticationStrategy(new AtLeastOneSuccessfulStrategy());
        return modularRealmAuthenticator;
    }
    @Bean
    public AdminRealm adminRealm() {
        AdminRealm adminRealm = new AdminRealm();
        return adminRealm;
    }
    @Bean
    public StaffRealm staffRealm() {
        StaffRealm staffRealm = new StaffRealm();
        return staffRealm;
    }
    @Bean
    public CodeRealm codeRealm() {
        CodeRealm adminShiroRealm = new CodeRealm();
        return adminShiroRealm;
    }
    @Bean
    public UserRealm userRealm() {
        UserRealm userRealm = new UserRealm();
        return userRealm;
    }


    /**
     * 开启shrio注解
     */
    @Bean
    public AuthorizationAttributeSourceAdvisor authorizationAttributeSourceAdvisor(DefaultWebSecurityManager securityManager) {
        AuthorizationAttributeSourceAdvisor advisor = new AuthorizationAttributeSourceAdvisor();
        advisor.setSecurityManager(securityManager);
        return advisor;
    }

    /**
     *自定义sessionManager
     */
    @Bean
    public MySessionManager sessionManager() {
        MySessionManager mySessionManager = new MySessionManager();
        mySessionManager.setSessionDAO(redisSessionDAO());
        return mySessionManager;
    }


    /**
     * 配置shiro redisManager
     * 使用的是shiro-redis开源插件
     */
    public IRedisManager redisManager() {
        RedisManager redisManager = new RedisManager();
        redisManager.setHost(host);
        redisManager.setPort(port);
        redisManager.setPassword(password);
        redisManager.setDatabase(5);
        return redisManager;
    }
    /**
     * cacheManager 缓存 redis实现
     * 使用的是shiro-redis开源插件
     */
     @Bean
     public RedisCacheManager cacheManagers() {
        RedisCacheManager redisCacheManager = new RedisCacheManager();
        redisCacheManager.setRedisManager(redisManager());
        redisCacheManager.setExpire(expireTime);

        return redisCacheManager;
    }

    /**
     * RedisSessionDAO shiro sessionDao层的实现 通过redis
     * SessionDAO的作用是为Session提供CRUD并进行持久化的一个shiro组件
     * MemorySessionDAO 直接在内存中进行会话维护
     * EnterpriseCacheSessionDAO  提供了缓存功能的会话维护，默认情况下使用MapCache实现，内部使用ConcurrentHashMap保存缓存的会话。
     */
    @Bean
    public RedisSessionDAO redisSessionDAO() {
        RedisSessionDAO redisSessionDAO = new RedisSessionDAO();
        redisSessionDAO.setRedisManager(redisManager());
        redisSessionDAO.setExpire(expireTime);
        return redisSessionDAO;
    }

    /**
     * 限制同一账号登录同时登录人数控制
     *
     * @return
     */
    @Bean
    public KickoutSessionControlFilter kickoutSessionControlFilter() {
        KickoutSessionControlFilter kickoutSessionControlFilter = new KickoutSessionControlFilter();
        kickoutSessionControlFilter.setCacheManager(cacheManagers());
        kickoutSessionControlFilter.setSessionManager(sessionManager());
        kickoutSessionControlFilter.setKickoutAfter(false);
        kickoutSessionControlFilter.setMaxSession(1);
        return kickoutSessionControlFilter;
    }


}
