package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.utils.R;
import cn.wisenergy.mapper.AdmissionRuleMapper;
import cn.wisenergy.mapper.ProfessionMapper;
import cn.wisenergy.mapper.ScoreInfoMapper;
import cn.wisenergy.model.app.AdmissionRule;
import cn.wisenergy.model.app.Profession;
import cn.wisenergy.model.app.ScoreInfo;
import cn.wisenergy.model.enums.SchemeTypeEnums;
import cn.wisenergy.model.enums.StudentType;
import cn.wisenergy.service.app.ProfessionService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * @author 86187
 * @ Description: 专业接口实现
 * @ Author     : 86187
 * @ Date       : 2021/1/13 10:52
 */
@Service
@Slf4j
public class ProfessionServiceImpl extends ServiceImpl<ProfessionMapper, Profession> implements ProfessionService {

    @Autowired
    private ProfessionMapper professionMapper;

    @Autowired
    private ScoreInfoMapper scoreInfoMapper;

    @Autowired
    private AdmissionRuleMapper admissionRuleMapper;

    @Override
    public R<List<Profession>> getTree(Integer type, Integer userId) {
        log.info("volunteer-service[]ProfessionServiceImpl[]getTree[]input.method");
        if (null == type || null == userId) {
            return R.error("入参为空!");
        }

        //获取方案类型
        Integer schemeType = getSchemeType(type, userId);
        if (null == schemeType) {
            return R.error("考生成绩信息不存在!");
        }

        //获取所有专业
        List<Profession> rootList = professionMapper.getList(schemeType);

        //构造树形接口：递归
        List<Profession> streetList = getStreetList(rootList);

        return R.ok(streetList);
    }

    private List<Profession> getStreetList(List<Profession> list) {
        List<Profession> rootList = new ArrayList<>();
        List<Profession> childrenList = new ArrayList<>();

        //1|遍历出根节点和子节点
        for (Profession profession : list) {
            if (profession.getLevel() == 1) {
                rootList.add(profession);
            } else {
                childrenList.add(profession);
            }
        }

        //2.遍历集合，获取根节点下的子节点
        for (Profession profession : rootList) {
            List<Profession> children = setChildren(profession.getId(), childrenList);
            profession.setChildren(children);
        }
        return rootList;
    }

    private List<Profession> setChildren(Integer id, List<Profession> list) {
        List<Profession> children = new ArrayList<>();
        for (Profession profession : list) {
            if (id.equals(profession.getParentId())) {
                children.add(profession);
            }
        }

        for (int i = 0; i < children.size(); i++) {
            Integer autoId = children.get(i).getId();
            List<Profession> childrenList = setChildren(autoId, list);
            children.get(i).setChildren(childrenList);
        }
        return children;
    }


    private Integer getSchemeType(Integer type, Integer userId) {
        //获取用户成绩
        ScoreInfo scoreInfo = scoreInfoMapper.getByUserId(userId);
        if (null == scoreInfo) {
            return null;
        }

        //type==1 文化生
        if (StudentType.CULTURE_STUDENT.getCode().equals(type)) {
            //获取文化课本科一批规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode());
            if (null == firstRule) {
                return null;
            }

            //本科类比文化分 = 本年录取分 -去年录取分 + 考生分数
            double culture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + scoreInfo.getCultureGrade();
            if (culture >= firstRule.getCultureMin()) {
                return SchemeTypeEnums.UNDERGRADUATE_CULTURE.getCode();
            }

            return SchemeTypeEnums.JUNIOR_COLLEGE_MAJOR.getCode();
        }

        //type == 2 美术考生
        if (StudentType.ART_STUDENT.getCode().equals(type)) {
            //获取美术本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode());
            if (null == firstRule) {
                return null;
            }

            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double trueCulture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + scoreInfo.getCultureGrade();
            double trueMajor = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + scoreInfo.getMajorGrade();
            if (trueCulture >= firstRule.getCultureMin() &&
                    trueMajor >= firstRule.getProfessionMin()) {
                return SchemeTypeEnums.UNDERGRADUATE_ARTS.getCode();
            }

            return SchemeTypeEnums.JUNIOR_COLLEGE_ARTS.getCode();
        }

        //3 体育考生
        if (StudentType.SPORTS_STUDENT.getCode().equals(type)) {
            //获取体育本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode());
            if (null == firstRule) {
                return null;
            }

            //计算综合成绩
            //综合成绩差值
            double value = firstRule.getCurrentYearCulture() - firstRule.getCultureMin();
            double total = scoreInfo.getMajorGrade() * 750 / 100 * 0.7 + scoreInfo.getCultureGrade() * 0.3 - value;
            if (total >= firstRule.getCultureMin()) {
                return SchemeTypeEnums.UNDERGRADUATE_SPORTS.getCode();
            }
            return SchemeTypeEnums.JUNIOR_COLLEGE_SPORTS.getCode();
        }

        //type == 4 文学编导
        if (StudentType.LITERATURE_STUDENT.getCode().equals(type)) {
            //获取文学编导本科一批录取规则
            AdmissionRule firstRule = admissionRuleMapper.getByType(SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode());
            if (null == firstRule) {
                return null;
            }

            //专科类比分 = 本年录取分 -去年录取分 + 考生分数
            double trueCulture = firstRule.getCurrentYearCulture() - firstRule.getCultureMin() + scoreInfo.getCultureGrade();
            double trueMajor = firstRule.getCurrentYearMajor() - firstRule.getProfessionMin() + scoreInfo.getMajorGrade();
            if (trueCulture >= firstRule.getCultureMin() && trueMajor >= firstRule.getProfessionMin()) {
                return SchemeTypeEnums.UNDERGRADUATE_LITERATURE.getCode();
            }

            return SchemeTypeEnums.JUNIOR_COLLEGE_LITERATURE.getCode();
        }
        return null;
    }
}
