package cn.wisenergy.service.app.impl;

import cn.wisenergy.common.constant.CommonAttributes;
import cn.wisenergy.common.utils.DateUtil;
import cn.wisenergy.common.utils.R;
import cn.wisenergy.mapper.SchemeRecordMapper;
import cn.wisenergy.mapper.UserVolunteerMapper;
import cn.wisenergy.mapper.UsersMapper;
import cn.wisenergy.mapper.VolunteerMapper;
import cn.wisenergy.model.app.SchemeQueryRecord;
import cn.wisenergy.model.app.User;
import cn.wisenergy.model.app.UserVolunteer;
import cn.wisenergy.model.app.Volunteer;
import cn.wisenergy.model.dto.VolunteerDto;
import cn.wisenergy.model.vo.SchemeRecordQueryVo;
import cn.wisenergy.service.app.SchemeRecordService;
import cn.wisenergy.service.common.PdfUtil;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.github.pagehelper.PageInfo;
import com.itextpdf.text.Document;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.pdf.PdfWriter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;


import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @author 86187
 * @ Description: 方案查询记录接口实现
 * @ Author     : 86187
 * @ Date       : 2021/1/8 11:41
 */
@Slf4j
@Service
public class SchemeRecordServiceImpl extends ServiceImpl<SchemeRecordMapper, SchemeQueryRecord> implements SchemeRecordService {
    @Autowired
    private SchemeRecordMapper schemeRecordMapper;

    @Autowired
    private UserVolunteerMapper userVolunteerMapper;

    @Autowired
    private VolunteerMapper volunteerMapper;

    @Autowired
    private UsersMapper usersMapper;

    @Override
    public R<SchemeQueryRecord> add(SchemeQueryRecord scheme) {
        log.info("volunteer-service[]SchemeRecordServiceImpl[]getById[]input.param.scheme:" + scheme);
        if (null == scheme) {
            return R.error("入参为空!");
        }

        //保存记录信息
        int count = schemeRecordMapper.add(scheme);
        if (count == 0) {
            return R.error("添加方案记录失败");
        }
        return R.ok(scheme);
    }

    @Override
    public R<PageInfo<SchemeQueryRecord>> getList(SchemeRecordQueryVo queryVo) {
        log.info("volunteer-service[]SchemeRecordServiceImpl[]getList[]input.param.queryVo:" + queryVo);
        if (null == queryVo || null == queryVo.getUserId()) {
            return R.error("入参为空!");
        }

        pageHandle(queryVo);

        //统计数据总条数
        Map<String, Object> map = new HashMap<>(8);
        map.put("userId", queryVo.getUserId());
        int total = schemeRecordMapper.count(map);


        //根据用户id,查询方案记录列表
        List<SchemeQueryRecord> list = schemeRecordMapper.getByUserId(queryVo.getUserId(),
                queryVo.getStartNum(), queryVo.getEndNum());
        PageInfo<SchemeQueryRecord> info = new PageInfo<>();
        info.setPageSize(queryVo.getPageSize());
        info.setPageNum(queryVo.getPageNo());
        info.setTotal(total);
        info.setList(list);
        return R.ok(info);
    }

    @Override
    public R<List<Volunteer>> getVolunteerList(Integer userId, Integer recordId) {
        log.info("volunteer-service[]SchemeRecordServiceImpl[]getVolunteerList[]input.param.userId,recordId:" + userId);
        if (null == userId || null == recordId) {
            return R.error("入参为空!");
        }

        //1、获取方案查询记录
        SchemeQueryRecord schemeQueryRecord = schemeRecordMapper.selectById(recordId);
        if (null == schemeQueryRecord) {
            return R.error("方案查询记录信息不存在!");
        }

        //2、根据用户id和方案记录id,获取志愿ids
        QueryWrapper<UserVolunteer> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", userId);
        queryWrapper.eq("scheme_record_id", recordId);
        List<UserVolunteer> list = userVolunteerMapper.selectList(queryWrapper);
        if (CollectionUtils.isEmpty(list)) {
            return R.ok(new ArrayList<>());
        }

        List<Integer> ids = list.stream().map(UserVolunteer::getVolunteerId).collect(Collectors.toList());
        List<Volunteer> result = volunteerMapper.getListByIds(ids);
        for (Volunteer volunteer : result) {
            volunteer.setQueryTime(schemeQueryRecord.getCreateTime());
        }
        result.sort(Comparator.comparing(Volunteer::getLowestMark).reversed());
        return R.ok(result);
    }

    /**
     * 方案导出
     *
     * @param userId   用户id
     * @param recordId 方案查询记录id
     * @return
     */
    @Override
    public R<Boolean> export(Integer userId, Integer recordId, HttpServletResponse response) {
        log.info("volunteer-service[]SchemeRecordServiceImpl[]export[]input.param.userId,recordId:" + userId);

        //查询时间
        String format = null;

        if (null == userId || null == recordId) {
            return R.error("入参为空!");
        }

        //获取用户信息
        User byId = usersMapper.getById(userId);
        String name = byId.getUserName();

        //1、根据用户id和方案记录id,获取志愿ids
        QueryWrapper<UserVolunteer> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", userId);
        queryWrapper.eq("scheme_record_id", recordId);
        List<UserVolunteer> list = userVolunteerMapper.selectList(queryWrapper);
        SchemeQueryRecord schemeQueryRecord = schemeRecordMapper.getById(recordId);
        if (CollectionUtils.isEmpty(list)) {
            return R.error("数据为空");
        }

        //获取方案查询时间
        if (null != schemeQueryRecord) {
            format = DateUtil.convertDateToStr(schemeQueryRecord.getUpdateTime(), "yyyy-MM-dd HH:mm:SS");
        } else {
            format = DateUtil.convertDateToStr(new Date(), "yyyy-MM-dd HH:mm:SS");
        }


        List<Integer> ids = list.stream().map(UserVolunteer::getVolunteerId).collect(Collectors.toList());
        List<Volunteer> result = volunteerMapper.getListByIds(ids);
        result.sort(Comparator.comparing(Volunteer::getLowestMark).reversed());
        //调用工具生成PDF
        try {
            // 1.新建document对象
            Document document = new Document(PageSize.A4);

            //创建文件名称
            Long lon = System.currentTimeMillis();

            // 2.建立一个书写器(Writer)与document对象关联
            OutputStream buffer = new ByteArrayOutputStream();
            PdfWriter writer = PdfWriter.getInstance(document, buffer);

            // 3.打开文档
            document.open();

            // 4.向文档中添加内容
            PdfUtil pdfUtil = new PdfUtil();
            pdfUtil.generatePDF(document, result, name, format);

            // 5.关闭文档
            document.close();

            //  设置内容格式 以及 编码方式
            response.setContentType("application/pdf");
            response.setCharacterEncoding("utf-8");
            response.setHeader("Content-disposition", "attachment;file=" + lon.toString() + ".pdf");

            DataOutput output = new DataOutputStream(response.getOutputStream());
            byte[] bytes = ((ByteArrayOutputStream) buffer).toByteArray();
            response.setContentLength(bytes.length);
            for (int i = 0; i < bytes.length; i++) {
                output.writeByte(bytes[i]);
            }

            ((DataOutputStream) output).flush();

            ((DataOutputStream) output).close();

        } catch (Exception e) {
            return R.ok(1, false);
        }
        return R.ok(0, true);
    }

    @Override
    public R<Boolean> createExcel(Integer userId, Integer recordId, HttpServletResponse response) {
        log.info("SchemeRecordServiceImpl[].createExcel[].input.param:userId,recordId" + userId, recordId);

        //查询时间
        String format = null;

        if (null == userId || null == recordId) {
            return R.error("入参为空!");
        }

        //获取用户信息
        User byId = usersMapper.getById(userId);

        if (null == byId) {
            return R.error("该用户不存在");
        }
        String name = byId.getUserName();

        //1、根据用户id和方案记录id,获取志愿ids
        QueryWrapper<UserVolunteer> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", userId);
        queryWrapper.eq("scheme_record_id", recordId);
        List<UserVolunteer> list = userVolunteerMapper.selectList(queryWrapper);
        SchemeQueryRecord schemeQueryRecord = schemeRecordMapper.getById(recordId);
        if (CollectionUtils.isEmpty(list)) {
            return R.error("数据为空");
        }

        //获取方案查询时间
        if (null != schemeQueryRecord) {
            format = DateUtil.convertDateToStr(schemeQueryRecord.getUpdateTime(), "yyyy-MM-dd HH:mm:SS");
        } else {
            format = DateUtil.convertDateToStr(new Date(), "yyyy-MM-dd HH:mm:SS");
        }
        List<Integer> ids = list.stream().map(UserVolunteer::getVolunteerId).collect(Collectors.toList());
        List<Volunteer> result = volunteerMapper.getListByIds(ids);
        result.sort(Comparator.comparing(Volunteer::getLowestMark).reversed());

        //生成Excel
        try {
            //  设置内容格式 以及 编码方式
            response.setContentType("application/vnd.ms-excel");
            response.setCharacterEncoding("utf-8");

            //  使用java8新特性的stream流去处理数据，把空的数据过滤掉
            List<VolunteerDto> resultBo = result.stream().filter(Objects::nonNull)
                    .map(t -> {
                        return VolunteerDto.builder()
                                .majorName(t.getMajorName())
                                .academy(t.getAcademy())
                                .courseDemand(t.getCourseDemand())
                                .nature(t.getNature())
                                .yearLimit(t.getYearLimit())
                                .planNum(t.getPlanNum())
                                .castArchivesNum(null == t.getCastArchivesNum() ? 0 : t.getCastArchivesNum())
                                .launchNum(null == t.getLaunchNum() ? 0 : t.getLaunchNum())
                                .lowestMark(null == t.getLowestMark() ? 0 : t.getLowestMark())
                                .lowestRank(null == t.getLowestRank() ? String.valueOf(0) : t.getLowestRank())
                                .build();
                    }).collect(Collectors.toList());

            //创建文件名称
            Long lon = System.currentTimeMillis();

            response.setHeader("Content-disposition", "attachment;filename=" + lon + ".xlsx");

            //  sheet名称
            EasyExcel.write(response.getOutputStream(), VolunteerDto.class).sheet(lon.toString()).doWrite(resultBo);
            return R.ok(0, true);
        } catch (Exception e) {
            return R.ok(1, false);
        }
    }


    /**
     * 分页处理
     *
     * @param cameraQueryVo 参数
     */
    private void pageHandle(SchemeRecordQueryVo cameraQueryVo) {
        Integer pageNum = cameraQueryVo.getPageNo();
        Integer pageSize = cameraQueryVo.getPageSize();
        if (null == pageSize || pageSize == 0) {
            pageSize = 10;
        }
        if (null == pageNum || pageNum == 0) {
            pageNum = 1;
        }
        Integer endNum = pageSize;
        Integer startNum = (pageNum - CommonAttributes.NUM_ONE) * pageSize;
        cameraQueryVo.setEndNum(endNum);
        cameraQueryVo.setStartNum(startNum);
        cameraQueryVo.setPageNo(pageNum);
        cameraQueryVo.setPageSize(pageSize);
    }
}
