package cn.wisenergy.web.admin.controller.app;

import cn.wisenergy.common.constant.RedisConsts;
import cn.wisenergy.common.utils.DateUtil;
import cn.wisenergy.common.utils.R;
import cn.wisenergy.model.app.User;
import cn.wisenergy.model.dto.UserCommitDto;
import cn.wisenergy.model.dto.UserInfoDto;
import cn.wisenergy.model.vo.UserInfoVo;
import cn.wisenergy.model.vo.UserQueryVo;
import cn.wisenergy.model.vo.UserShowVo;
import cn.wisenergy.model.vo.UserVipCommitVo;
import cn.wisenergy.service.app.UserService;
import cn.wisenergy.web.config.JwtConfig;
import cn.wisenergy.web.shiro.JwtUtil;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;


/**
 * @ Description: 用户管理模
 * @ Author     : 86187
 * @ Date       : 2021/1/7 10:29
 * @author 86187
 */
@RestController
@Api(tags = "PC-用户管理")
@RequestMapping("/user")
@Slf4j
public class UserController {
    @Autowired
    private UserService userService;

    @Autowired
    private JwtUtil jwtUtil;
    @Autowired
    private JwtConfig jwtConfig;
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;


    @ApiOperation(value = "用户管理", notes = "用户管理", httpMethod = "POST")
    @ApiImplicitParam(name = "queryVo", value = "用户信息", dataType = "UserQueryVo")
    @PostMapping("/manage")
    public R<PageInfo<UserInfoDto>> manageUser(@RequestBody UserQueryVo queryVo) {
        log.info("UserController[]manageUser[]input.param:queryV0:" + queryVo);
        if (null == queryVo) {
            return R.error("入参为空!");
        }
        //返回数据
        return userService.getUserList(queryVo);
    }

    @ApiOperation(value = "提交个人信息", notes = "提交个人信息", httpMethod = "POST")
    @ApiImplicitParam(name = "userInfo", value = "个人信息", dataType = "UserCommitDto")
    @PostMapping("/commit")
    public R<UserInfoVo> commitUserInfo(@RequestBody UserCommitDto userInfo) {
        log.info("UserController[]commitUserInfo[]input.param:queryV0:" + userInfo);
        if (null == userInfo) {
            return R.error("入参为空!");
        }
        //返回数据
        return userService.commitUserInfo(userInfo);
    }

    @ApiOperation(value = "编辑个人信息", notes = "编辑个人信息", httpMethod = "POST")
    @ApiImplicitParam(name = "userInfo", value = "个人信息", dataType = "UserCommitDto")
    @PostMapping("/edit")
    public R<Boolean> edit(@RequestBody UserCommitDto userInfo) {
        log.info("UserController[]edit[]input.param:queryV0:" + userInfo);
        if (null == userInfo) {
            return R.error("入参为空!");
        }
        //返回数据
        return userService.edit(userInfo);
    }

    @ApiOperation(value = "获取个人信息", notes = "获取个人信息", httpMethod = "GET")
    @ApiImplicitParam(name = "userId", value = "用户id", dataType = "int")
    @GetMapping("/getById")
    public R<UserShowVo> getById(Integer userId) {
        log.info("UserController[]getById[]input.param:queryV0:" + userId);
        if (null == userId) {
            return R.error("入参为空!");
        }

        //返回数据
        return userService.getById(userId);
    }

    @ApiOperation(value = "测试like循环查询", notes = "测试like循环查询", httpMethod = "GET")
    @GetMapping("/test")
    public R<List<User>> test() {
        log.info("UserController[]test[]input.param");

        //返回数据
        return userService.test();
    }

    /**
     * shiro登录
     *
     * @return
     */
    @ApiOperation(value = "获取token接口", notes = "获取token接口", httpMethod = "POST")
    @PostMapping(value = "/login")
    public R<String> login(String phone) {
        if (StringUtils.isBlank(phone)) {
            return R.error("入参为空!");
        }

        //用户信息
        User user = userService.getByPhone(phone);

        // 创建token
        String token = jwtUtil.generateToken(user);
        // 保存Redis
        redisTemplate.opsForValue().set(RedisConsts.JWT_ACCESS_TOKEN + token, token);
        return R.ok(token);
    }

    @ApiOperation(value = "用户Excel导出",notes = "用户Excel导出",httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userName",value = "用户名",dataType="string"),
            @ApiImplicitParam(name = "pageSize",value = "电话",dataType="string"),
            @ApiImplicitParam(name = "startTime",value = "开始时间",dataType="string"),
            @ApiImplicitParam(name = "endTime",value = "结束时间",dataType="string")
    })
    @GetMapping(value = "/createExcel")
    public  R<Boolean> createExcel( String userName,String phone,String startTime,String endTime, HttpServletResponse response){
        log.info("UserController[]createExcel[]input.param:" + userName,phone,startTime,endTime);

        UserQueryVo userQueryVo = new UserQueryVo();

        if (StringUtils.isEmpty(userName)){
            userQueryVo.setUserName(null);
        }else {
            userQueryVo.setUserName(userName);
        }

        if (StringUtils.isEmpty(phone)) {
            userQueryVo.setPhone(null);
        }else{
            userQueryVo.setPhone(phone);
        }

        if (StringUtils.isEmpty(endTime)) {
            userQueryVo.setEndTime(null);
        }else{
            userQueryVo.setEndTime(DateUtil.getTime(endTime,"yyyy-MM-dd HH:mm:ss"));
        }

        if (StringUtils.isEmpty(startTime)) {
            userQueryVo.setStartTime(null);
        }else{
            userQueryVo.setStartTime(DateUtil.getTime(startTime,"yyyy-MM-dd HH:mm:ss"));
        }

        return userService.createExcel(userQueryVo,response);
    }

    @ApiOperation(value = "添加Vip客户",notes = "添加Vip客户",httpMethod = "POST")
    @ApiModelProperty(name = "userVipCommitVo",value = "vip客户信息",dataType = "UserVipCommitVo")
    @PostMapping("/add")
    public R<UserInfoVo> addUserVip(@RequestBody UserVipCommitVo userVipCommitVo){
        log.info("UserController[].addUserVip[].input.param:"+userVipCommitVo);
        return userService.commitUserVipInfo(userVipCommitVo);
    }


    @ApiOperation(value = "用户登录小程序授权",notes = "用户登录小程序授权",httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code",value = "用户临时code",dataType="string"),
            @ApiImplicitParam(name = "userId",value = "用户userId",dataType="Integer"),
    })
    @GetMapping("/appletBinding")
    public R<String> appletBinding(String code,Integer userId){
        log.info("UserController[]appletBinding[]input.param:queryV0:" + code);
        return userService.appletBinding(code,userId);
    }

    @ApiOperation(value = "查看用户是否授权",notes = "查看用户是否授权",httpMethod = "GET")
    @ApiImplicitParam(name = "code",value = "用户临时code",dataType="string")
    @GetMapping("/getByOpenid")
    public R<UserInfoVo> getByOpenid(String code){
        log.info("UserController[]getByUnionid[]input.param:queryV0:" + code);
        return userService.getByOpenid(code);
    }

    @ApiOperation(value = "退出小程序",notes = "退出小程序",httpMethod = "GET")
    @ApiImplicitParam(name = "userId",value = "用户userId",dataType="Integer")
    @GetMapping("/quitApplet")
    public R<String> quitApplet(Integer userId){
        log.info("UserController[]quitApplet[]input.param:queryV0:" + userId);
        return userService.quitApplet(userId);
    }

}
