package cn.wisenergy.web.shiro;

import cn.wisenergy.common.enums.RespCodeEnum;
import cn.wisenergy.model.app.User;
import cn.wisenergy.web.config.JwtConfig;
import com.alibaba.fastjson.JSON;
import io.jsonwebtoken.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authc.IncorrectCredentialsException;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * Jwt-Token工具类
 *
 * @author lut
 */
@Slf4j
@Component
@EnableConfigurationProperties(JwtConfig.class)
public class JwtUtil {

    @Autowired
    JwtConfig jwtConfig;

    /**
     * 生成Jwt令牌
     *
     * @param jSubject
     * @return
     */
    private String createJWT(String jSubject) {
        log.info("构建Jwt令牌-->{}", jSubject);
        return Jwts.builder()
                .signWith(SignatureAlgorithm.HS512, jwtConfig.getSecret())
                .setSubject(jSubject)
                .setIssuedAt(DateTime.now().toDate())
                //.setExpiration(DateTime.now().plusSeconds(jwtConfig.getExpire() * 2).toDate())
                .compact();
    }

    /**
     * 解析令牌数据
     *
     * @param token
     * @return
     */
    public Claims getClaimsFromToken(String token) {
        Claims claims;
        try {
            claims = Jwts.parser()
                    .setSigningKey(jwtConfig.getSecret())
                    .parseClaimsJws(token)
                    .getBody();
        } catch (ExpiredJwtException e) {
            throw new IncorrectCredentialsException(RespCodeEnum.TOKEN_IS_NOT_TIMEOUT.getMsg());
        } catch (MalformedJwtException | SignatureException e) {
            log.error("JWT解析异常，ex={}", e.getMessage(), e);
            throw new IncorrectCredentialsException(RespCodeEnum.TOKEN_IS_NOT_ERROR.getMsg());
        }
        return claims;
    }

    /**
     * 生成Jwt令牌
     *
     * @param userDetails 用户信息
     * @return
     */
    public String generateToken(User userDetails) {

        return createJWT(JSON.toJSONString(userDetails));
    }

    /**
     * 从令牌中获取用户信息
     *
     * @return
     */
    public User getUserFromToken(Claims claims) {

        return JSON.parseObject(claims.getSubject(), User.class);
    }

    /**
     * 判断令牌是否过期
     *
     * @param claims
     * @return 是否过期
     */
    public Boolean isTokenExpired(Claims claims) {
        return claims.getExpiration().before(DateTime.now().plusSeconds(jwtConfig.getExpire()).toDate());
    }

    /**
     * 从请求头中获取token
     *
     * @return
     */
    public String getToken() {
        HttpServletRequest request = ((ServletRequestAttributes) (RequestContextHolder.currentRequestAttributes())).getRequest();
        return request.getHeader("Authorization");
    }
}
