package cn.wise.sc.acquisition.business.util;

import org.apache.commons.lang3.StringUtils;

import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.time.temporal.WeekFields;
import java.util.Date;

/**
 * @author neo.shu
 * @since 2020/4/5 22:26
 */

public class DateUtil {

    public static final String FORMAT_TO_DAY = "yyyy-MM-dd";

    public static final String FORMAT_TO_MONTH = "yyyy-MM";

    public static final String FORMAT_DAY_TO_SECOND = " HH:mm:ss";

    public static final String FORMAT_TO_SECOND = "yyyy-MM-dd HH:mm:ss";

    private static final int ONE = 1;

    /**
     * 获取自然周数
     *
     * @param localDateTime
     * @return
     */
    public static int getWeek(LocalDateTime localDateTime) {
        WeekFields weekFields = WeekFields.ISO;
        return localDateTime.get(weekFields.weekOfWeekBasedYear());
    }

    /**
     * 获取开始时间
     *
     * @param type 类型 1:本周 2:本月 3:本年
     * @return 开始时间
     */
    public static Date getStartTime(Integer type) {

        Date rts = null;

        if (type == 1) {
            rts = cn.hutool.core.date.DateUtil.beginOfWeek(cn.hutool.core.date.DateUtil.date());
        } else if (type == 2) {
            rts = cn.hutool.core.date.DateUtil.beginOfMonth(cn.hutool.core.date.DateUtil.date());
        } else {
            rts = cn.hutool.core.date.DateUtil.beginOfYear(cn.hutool.core.date.DateUtil.date());
        }

        return rts;
    }


    /**
     * 获取今天  例如参数为 2021-04-05 17:37:33  返回 2021-04-05 00.00.00
     *
     * @param localDateTime
     * @return
     */
    public static LocalDateTime getToday(LocalDateTime localDateTime, String format) {
        return formatDateToTime(localDateTime, format);
    }



    /**
     * 获取下一天  例如参数为 2021-04-05 17:37:33  返回 2021-04-06 00.00.00
     *
     * @param localDateTime
     * @return
     */
    public static LocalDateTime getTomorrow(LocalDateTime localDateTime, String format) {
        return formatDateToTime(localDateTime.plusDays(ONE), format);
    }




    /**
     * 按照参数格式化时间
     *
     * @param localDateTime
     * @return
     */
    private static LocalDateTime formatDateToTime(LocalDateTime localDateTime, String format) {
        if (localDateTime == null) throw new RuntimeException("formatDate method:LocalDateTime localDateTime is null");
        if (StringUtils.isBlank(format)) throw new RuntimeException("formatDate method:String format is null");

        DateTimeFormatter sdf = DateTimeFormatter.ofPattern(format);
        String date = sdf.format(localDateTime);
        return LocalDateTime.parse(date);
    }



    /**
     * 获取今天的最后一刻
     */
    public static String getEndTimeStr(LocalDateTime localDateTime) {
        LocalDateTime endT = getEndTime(localDateTime);//今天23点59分59秒的毫秒数
        String date = DateTimeFormatter.ofPattern(FORMAT_TO_SECOND).format(endT);
        return date;
    }

    /**
     * 获取今天的最后一刻
     */
    public static LocalDateTime getEndTime(LocalDateTime localDateTime) {
        if (localDateTime == null) throw new RuntimeException("getEndTime method:LocalDateTime localDateTime is null");
        LocalDateTime endT = LocalDateTime.of(localDateTime.toLocalDate(), LocalTime.MAX);//今天23点59分59秒的毫秒数
        return endT;
    }


    /**
     * 获取今天的00时刻
     */
    public static String getBeginTimeStr(LocalDateTime localDateTime) {
        LocalDateTime zeroT = getBeginTime(localDateTime);//今天零点零分零秒的毫秒数
        String date = DateTimeFormatter.ofPattern(FORMAT_TO_SECOND).format(zeroT);
        return date;
    }


    /**
     * 获取今天的00一刻
     */
    public static LocalDateTime getBeginTime(LocalDateTime localDateTime) {
        if (localDateTime == null) throw new RuntimeException("getEndTime method:LocalDateTime localDateTime is null");
        LocalDateTime endT = LocalDateTime.of(localDateTime.toLocalDate(), LocalTime.MIN);//今天23点59分59秒的毫秒数
        return endT;
    }

    /**
     * 字符串转化为时间
     */
    public static LocalDateTime strToTime(String timeStr){
        if (StringUtils.isBlank(timeStr)) throw new RuntimeException("LocalDateTime strToTime: String timeStr is null");
        DateTimeFormatter fmt = new DateTimeFormatterBuilder()
                .appendPattern(FORMAT_TO_DAY)
                .optionalStart()
                .appendPattern(FORMAT_DAY_TO_SECOND)
                .optionalEnd()
                .parseDefaulting(ChronoField.HOUR_OF_DAY, 0)
                .parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
                .toFormatter();
        try {
            LocalDateTime ldt = LocalDateTime.parse(timeStr,fmt);
            return ldt;
        }catch (Exception e){
            throw new RuntimeException("transform str to time is failed");
        }
    }


    /**
     * 时间转化为字符串
     */
    public static String timeToStr(LocalDateTime time,String format){
        if (time == null) throw new RuntimeException("LocalDateTime timeToStr: LocalDateTime time is null");
        try {
            String date = DateTimeFormatter.ofPattern(format).format(time);
            return date;
        }catch (Exception e){
            throw new RuntimeException("transform time to str is failed");
        }

    }
}
