package cn.wise.sc.cement.business.controller;

import cn.wise.sc.cement.business.model.BaseResponse;
import cn.wise.sc.cement.business.model.PageQuery;
import cn.wise.sc.cement.business.model.ReportDetailVo;
import cn.wise.sc.cement.business.model.SixElementKey;
import cn.wise.sc.cement.business.model.SixElementReport;
import cn.wise.sc.cement.business.model.vo.EntrustVo;
import cn.wise.sc.cement.business.service.IEntrustService;
import cn.wise.sc.cement.business.util.WordUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description: 报告管理
 * @author: qh
 * @create: 2020-09-25 14:37
 **/
@Api(tags = "报告管理")
@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/business/report")
public class ReportController {

    final
    IEntrustService iEntrustService;

    public ReportController(IEntrustService iEntrustService) {
        this.iEntrustService = iEntrustService;

    }

    @ApiOperation(value = "报告分页列表")
    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "startDate", value = "开始日期", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "endDate", value = "结束日期", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "clientId", value = "委托单位id", paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "projectName", value = "项目名称", paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "projectCode", value = "项目编号", paramType = "query", dataType = "String")
    })
    @GetMapping("/getPage")
    public BaseResponse<IPage<EntrustVo>> page(PageQuery pageQuery, String startDate, String endDate,
                                               Integer clientId, String projectName, String projectCode) {
        return iEntrustService.getReportPage(pageQuery, startDate, endDate, clientId, projectName, projectCode);
    }

    @ApiOperation(value = "置顶取消置顶 (createTime = updateTime显示置顶，否则显示取消置顶)")
    @PostMapping("/setTopping")
    public BaseResponse<String> setTopping(Integer id) {
        try {
            return iEntrustService.setTopping(id);
        } catch (Exception e) {
            log.debug("置顶取消置顶 {}", e.getMessage());
        }
        return BaseResponse.errorMsg("失败!");
    }

    @DeleteMapping("/{id}")
    @ApiOperation("删除报告")
    public BaseResponse<Boolean> deleteById(@PathVariable("id") Integer id) {
        return iEntrustService.deleteById(id);
    }

    @PostMapping("/{entrustId}")
    @ApiOperation("导出报告封面和首页")
    public void getReportDetail(@PathVariable("entrustId") Integer entrustId,
                                HttpServletResponse response) {

        //首页及封面导出
        ReportDetailVo rts = iEntrustService.getReportDetail(entrustId);
        Map<String, Object> beanParams = new HashMap<>(10);
        beanParams.put("sendName", rts.getSendName());
        beanParams.put("sender", rts.getSender());
        beanParams.put("sendDate", rts.getSendDate());
        beanParams.put("sampleNum", rts.getSampleNum());
        beanParams.put("sampleNames", set2String(rts.getSampleNames()));
        beanParams.put("teamNames", set2String(rts.getTeamNames()));
        beanParams.put("methodNames", set2StringN(rts.getMethodNames()));
        beanParams.put("equipmentNames", set2StringN(rts.getEquipmentNames()));
        beanParams.put("projectName", rts.getProjectName());
        beanParams.put("printDate", rts.getPrintDate());

        //六元素导出
        List<SixElementReport> list = iEntrustService.getSampleSixElementCheck(entrustId);
        list.forEach(this::initMapStr2AlongPro);
        List<SixElementReport> Al2O3AndTiO2List = list.stream()
                .filter(arg -> !"-".equals(arg.getAl2o3AndTio2()))
                .collect(Collectors.toList());
        List<SixElementReport> Al2O3SplitTiO2List = list.stream()
                .filter(arg -> "-".equals(arg.getAl2o3AndTio2()))
                .collect(Collectors.toList());
        beanParams.put("list1", Al2O3AndTiO2List);
        beanParams.put("list2", Al2O3SplitTiO2List);


        WordUtil.writeWordReport(rts.getProjectName() + "(报告).xls", "report.ftl",
                beanParams, response);

    }

    /**
     * 将样品六元素检测Map转换成单独属性
     *
     * @param sixElement 带转换的六元素检测结果
     * @return 已转换结果
     */
    private SixElementReport initMapStr2AlongPro(SixElementReport sixElement) {

        String countResult = sixElement.getCountResult();
        HashMap<String, String> countResultMap = JSON.parseObject(countResult, HashMap.class);

        sixElement.setAl2o3("-");
        sixElement.setTio2("-");
        sixElement.setAl2o3AndTio2("-");
        //判断检测结果中六元素Al2O3是否包含TiO2
        if (countResultMap.containsKey(SixElementKey.Al2O3AndTiO2.getKey())) {
            sixElement.setAl2o3AndTio2(countResultMap.get(SixElementKey.Al2O3AndTiO2.getKey()));
        } else {
            sixElement.setAl2o3(!countResultMap.containsKey(SixElementKey.Al2O3.getKey()) ? "0" : countResultMap.get(SixElementKey.Al2O3.getKey()));
            sixElement.setTio2(!countResultMap.containsKey(SixElementKey.TiO2.getKey()) ? "0" : countResultMap.get(SixElementKey.TiO2.getKey()));
        }

        sixElement.setCao(!countResultMap.containsKey(SixElementKey.CaO.getKey()) ? "0" : countResultMap.get(SixElementKey.CaO.getKey()));
        sixElement.setCl(!countResultMap.containsKey(SixElementKey.Cl.getKey()) ? "0" : countResultMap.get(SixElementKey.Cl.getKey()));
        sixElement.setFe2o3(!countResultMap.containsKey(SixElementKey.Fe2O3.getKey()) ? "0" : countResultMap.get(SixElementKey.Fe2O3.getKey()));
        sixElement.setLoi(!countResultMap.containsKey(SixElementKey.LOI.getKey()) ? "0" : countResultMap.get(SixElementKey.LOI.getKey()));
        sixElement.setMgo(!countResultMap.containsKey(SixElementKey.MgO.getKey()) ? "0" : countResultMap.get(SixElementKey.MgO.getKey()));
        sixElement.setSio2(!countResultMap.containsKey(SixElementKey.SiO2.getKey()) ? "0" : countResultMap.get(SixElementKey.SiO2.getKey()));
        sixElement.setSo3(!countResultMap.containsKey(SixElementKey.SO3.getKey()) ? "0" : countResultMap.get(SixElementKey.SO3.getKey()));

        return sixElement;
    }

    public static void main(String[] args) {
        String str = "{\"Al2O3+TiO2\":\"-1488450.00\",\"CaO\":\"3650.00\",\"MgO\":\"0.00\",\"L.O.I\":\"10218.00\",\"SiO2\":\"2750.00\",\"Al2O3\":\"-1902250.00\",\"Fe2O3\":\"2100.00\"}";
        Map<String, String> map = JSON.parseObject(str, HashMap.class);
        System.out.println(map.toString());
    }

    private String set2String(Set<String> source) {
        if (source.size() == 0) {
            return "";
        }
        StringBuilder strBuilder = new StringBuilder();
        for (String target : source) {
            strBuilder.append("、").append(target);
        }
        return strBuilder.replace(0, 1, "").toString();
    }

    private String set2StringN(Set<String> source) {
        if (source.size() == 0) {
            return "";
        }
        StringBuilder strBuilder = new StringBuilder();
        for (String target : source) {
            strBuilder.append("&#10;").append(target);
        }

        return strBuilder.replace(0, 5, "").toString();
    }
}
