package cn.wise.sc.cement.business.service.impl;

import cn.wise.sc.cement.business.entity.*;
import cn.wise.sc.cement.business.mapper.StandardEnterMapper;
import cn.wise.sc.cement.business.mapper.StandardMapper;
import cn.wise.sc.cement.business.mapper.StandardOutMapper;
import cn.wise.sc.cement.business.model.BaseResponse;
import cn.wise.sc.cement.business.model.LoginUser;
import cn.wise.sc.cement.business.model.PageQuery;
import cn.wise.sc.cement.business.model.query.*;
import cn.wise.sc.cement.business.model.vo.StandardValueVo;
import cn.wise.sc.cement.business.model.vo.StandardVo;
import cn.wise.sc.cement.business.service.IStandardEnterService;
import cn.wise.sc.cement.business.service.IStandardService;
import cn.wise.sc.cement.business.service.IStandardValueService;
import cn.wise.sc.cement.business.service.ISysUserService;
import cn.wise.sc.cement.business.util.ExcelUtil;
import cn.wise.sc.cement.business.wrapper.page.Query;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author ztw
 * @since 2020-09-04
 */
@Service
public class StandardServiceImpl extends ServiceImpl<StandardMapper, Standard> implements IStandardService {

    @Resource
    private StandardMapper standardMapper;
    @Autowired
    private ISysUserService userService;
    @Resource
    private StandardEnterMapper standardEnterMapper;
    @Resource
    private StandardOutMapper standardOutMapper;
    @Autowired
    private IStandardValueService standardValueService;
    @Autowired
    private CommonServiceImpl commonService;

    /**
     * 分页查询
     *
     * @param pageQuery
     * @param name
     * @param supplierName
     * @return
     */
    @Override
    public BaseResponse<IPage<Standard>> getPage(PageQuery pageQuery, String name, String supplierName) {
        QueryWrapper<Standard> qw = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(name)) {
            qw.like("name", name);
        }
        if (StringUtils.isNotEmpty(supplierName)) {
            qw.like("supplier_name", supplierName);
        }
        qw.orderByDesc("name");
        IPage<Standard> page = new Page<>(pageQuery.getPageNo(), pageQuery.getPageSize());
        page = standardMapper.selectPage(page, qw);
        return BaseResponse.okData(page);
    }

    /**
     * 标样信息列表导出
     * @param name
     * @param supplierName
     * @param fileName
     * @param response
     */
    @Override
    public void export(String name, String supplierName, String fileName, HttpServletResponse response) {
        Map<String, Object> params = new HashMap<>();
        params.put("supplierName", supplierName);
        params.put("name", name);
        List<Map<String, Object>> list= standardMapper.exportList(params);

        if (!CollectionUtils.isEmpty(list)) {
            Map<String, Object> map = list.get(0);
            String[] headers = new String[map.size()];
            headers[0] = "序号";
            headers[1] = "标样名称";
            headers[2] = "标物代号";
            headers[3] = "经销单位";
            headers[4] = "存放位置";
            headers[5] = "进样日期";
            headers[6] = "有效日期";
            headers[7] = "剩余库存";
            headers[8] = "状态";

            List<Object[]> datas = new ArrayList<>(list.size());
            for (Map<String, Object> m : list) {
                Object[] objects = new Object[headers.length];
                for (int j = 0; j < headers.length; j++) {
                    String obj = m.get(headers[j]).toString();
                    if(j==0){
                        obj = obj.split("\\.")[0];
                    }
                    objects[j] = obj;
                }
                datas.add(objects);
            }
            ExcelUtil.excelExport(
                    fileName == null || fileName.trim().length() <= 0 ? "标样信息": fileName, headers,
                    datas, response);
        }
    }


    /**
     * 新增标样
     *
     * @param query
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<Standard> create(StandardQuery query) {
        if (StringUtils.isEmpty(query.getName())) {
            return BaseResponse.errorMsg("标样名称不能为空!");
        }
        QueryWrapper<Standard> qw = new QueryWrapper<>();
        qw.eq("name", query.getName());
        int count = standardMapper.selectCount(qw);
        if (count > 0) {
            return BaseResponse.errorMsg(query.getName() + "已存在");
        }
        Standard standard = new Standard();
        BeanUtils.copyProperties(query, standard);
        standard.setStatus(1)
                .setCreateTime(LocalDateTime.now());
        standardMapper.insert(standard);
        //添加标准值
        if (query.getStandardValueQueryList() != null && query.getStandardValueQueryList().size() > 0) {
            List<StandardValueQuery> standardValueQueryList = query.getStandardValueQueryList();

            List<StandardValue> standardValueList = new ArrayList<>();
            for (StandardValueQuery standardValueQuery : standardValueQueryList) {
                StandardValue standardValue = new StandardValue();
                standardValue.setStandardId(standard.getId())
                        .setElementName(standardValueQuery.getElementName())
                        .setCreateTime(LocalDateTime.now());
                if (standardValueQuery.getElementValue() != null && standardValueQuery.getElementValue().size() > 0) {
                    String elementValue = JSON.toJSON(standardValueQuery.getElementValue()).toString();
                    standardValue.setElementValue(elementValue);
                } else {
                    standardValue.setElementValue(null);
                }
                standardValueList.add(standardValue);
            }
            standardValueService.saveBatch(standardValueList);
        }
        //标样入库记录添加
        LoginUser loginUser = userService.getLoginUser();
        StandardEnter enter = new StandardEnter();
        enter.setStandardId(standard.getId())
                .setUserId(loginUser.getId())
                .setEnterNum(query.getStockNum())
                .setCreateTime(LocalDateTime.now())
                .setRemark("新增标样");
        standardEnterMapper.insert(enter);
        //保存标样材料信息
        Boolean ref = commonService.saveEntityEnclosure(EntityEnclosure.EntityType.STANDARD_SAMPLE, standard.getId(), query.getEnclosureQueryList());
        if (!ref) {
            return BaseResponse.errorMsg("保存保存校核附件失败!");
        }

        return BaseResponse.okData(standard);
    }

    /**
     * 修改标样
     *
     * @param query
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<Standard> update(StandardQuery query) {
        if (StringUtils.isEmpty(query.getName())) {
            return BaseResponse.errorMsg("标样名称不能为空!");
        }
        QueryWrapper<Standard> qw = new QueryWrapper<>();
        qw.eq("name", query.getName());
        qw.ne("id", query.getId());
        int count = standardMapper.selectCount(qw);
        if (count > 0) {
            return BaseResponse.errorMsg(query.getName() + "已存在");
        }
        Standard standard = new Standard();
        BeanUtils.copyProperties(query, standard);
        standardMapper.updateById(standard);
        //添加标准值
        if (query.getStandardValueQueryList() != null && query.getStandardValueQueryList().size() > 0) {
            //删除之前的标准值信息
            QueryWrapper<StandardValue> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("standard_id", standard.getId());
            standardValueService.remove(queryWrapper);
            //添加新的标准值
            List<StandardValueQuery> standardValueQueryList = query.getStandardValueQueryList();
            List<StandardValue> standardValueList = new ArrayList<>();
            for (StandardValueQuery standardValueQuery : standardValueQueryList) {
                StandardValue standardValue = new StandardValue();
                standardValue.setStandardId(standard.getId())
                        .setElementName(standardValueQuery.getElementName())
                        .setCreateTime(LocalDateTime.now());
                if (standardValueQuery.getElementValue() != null && standardValueQuery.getElementValue().size() > 0) {
                    String elementValue = JSON.toJSON(standardValueQuery.getElementValue()).toString();
                    standardValue.setElementValue(elementValue);
                } else {
                    standardValue.setElementValue(null);
                }
                standardValueList.add(standardValue);
            }
            standardValueService.saveBatch(standardValueList);
        }
        //保存标样材料信息
        Boolean ref = commonService.saveEntityEnclosure(EntityEnclosure.EntityType.STANDARD_SAMPLE, standard.getId(), query.getEnclosureQueryList());
        if (!ref) {
            return BaseResponse.errorMsg("保存保存校核附件失败!");
        }
        return BaseResponse.okData(standard);
    }

    /**
     * 标样详情
     *
     * @param id 标样表id
     * @return
     */
    @Override
    public BaseResponse<StandardVo> getDetail(Integer id) {
        Standard standard = standardMapper.selectById(id);
        if (standard == null) {
            return BaseResponse.errorMsg("信息错误!");
        }
        StandardVo standardVo = new StandardVo();
        BeanUtils.copyProperties(standard, standardVo);
        List<EntityEnclosure> entityEnclosureList = commonService.getEnclosureList
                (EntityEnclosure.EntityType.STANDARD_SAMPLE, standard.getId());

        List<EnclosureQuery> enclosureQueryList = new ArrayList<>();
        for (EntityEnclosure entityEnclosure : entityEnclosureList) {
            EnclosureQuery enclosureQuery = new EnclosureQuery();
            enclosureQuery.setAlias(entityEnclosure.getAlias())
                    .setEnclosureUrl(entityEnclosure.getEnclosureUrl())
                    .setExtName(entityEnclosure.getExtName());
            enclosureQueryList.add(enclosureQuery);
        }
        standardVo.setEnclosureQueryList(enclosureQueryList);

        QueryWrapper<StandardValue> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("standard_id", standard.getId());
        List<StandardValue> standardValueList = standardValueService.list(queryWrapper);
        standardVo.setStandardValueList(standardValueList);
        return BaseResponse.okData(standardVo);
    }


    /**
     * 新增入库
     *
     * @param query
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<String> enter(ChangeStockQuery query) {
        if (query == null || query.getId() == null) {
            return BaseResponse.errorMsg("参数错误!");
        }
        Standard standard = standardMapper.selectById(query.getId());
        if (standard == null) {
            return BaseResponse.errorMsg("数据错误!");
        }
        StandardEnter enter = new StandardEnter();
        enter.setStandardId(standard.getId())
                .setUserId(userService.getLoginUser().getId())
                .setEnterNum(query.getChangeNum())
                .setCreateTime(LocalDateTime.now())
                .setRemark(query.getRemark());

        standardEnterMapper.insert(enter);
        standard.setStockNum(standard.getStockNum().add(enter.getEnterNum()));
        standardMapper.updateById(standard);
        return BaseResponse.okData("新增入库完成");
    }

    /**
     * 领用出库
     *
     * @param query
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<String> out(ChangeStockQuery query) {
        if (query == null || query.getId() == null) {
            return BaseResponse.errorMsg("参数错误!");
        }
        Standard standard = standardMapper.selectById(query.getId());
        if (standard == null) {
            return BaseResponse.errorMsg("数据错误!");
        }
        StandardOut out = new StandardOut();
        out.setStandardId(standard.getId())
                .setUserId(userService.getLoginUser().getId())
                .setCollectNum(query.getChangeNum())
                .setCreateTime(LocalDateTime.now())
                .setRemark(query.getRemark());
        standardOutMapper.insert(out);
        standard.setStockNum(standard.getStockNum().subtract(out.getCollectNum()));
        standardMapper.updateById(standard);
        return BaseResponse.okData("领用出库完成");
    }

    /**
     * 变更有效期
     *
     * @param query
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<String> changeValidDate(ChangeValidDateQuery query) {
        if (query == null || query.getId() == null) {
            return BaseResponse.errorMsg("参数错误!");
        }
        Standard standard = standardMapper.selectById(query.getId());
        if (standard == null) {
            return BaseResponse.errorMsg("数据错误!");
        }

        standard.setValidDate(query.getValidDate());
        standardMapper.updateById(standard);
        return BaseResponse.okData("变更有效期完成");
    }


    /**
     * 过期处置
     *
     * @param id
     * @return
     */
    @Transactional
    @Override
    public BaseResponse<String> overdue(Integer id) {
        if (id == null) {
            return BaseResponse.errorMsg("参数错误!");
        }
        Standard standard = standardMapper.selectById(id);
        if (standard == null) {
            return BaseResponse.errorMsg("数据错误!");
        }
        standard.setStatus(0);
        standardMapper.updateById(standard);
        return BaseResponse.okData("过期处置完成");
    }

    @Override
    public BaseResponse<List<StandardValueVo>> getStandardDetailList() {

        List<StandardValueVo> standardValueVos = standardMapper.getStandardDetaulList();
        if (standardValueVos.size() != 0) {
            return BaseResponse.okData(standardValueVos);
        } else {
            return BaseResponse.errorMsg("没找数据!");
        }
    }


}
