package cn.wise.sc.cement.business.service.impl;

import cn.wise.sc.cement.business.entity.*;
import cn.wise.sc.cement.business.mapper.*;
import cn.wise.sc.cement.business.model.BaseResponse;
import cn.wise.sc.cement.business.model.PageQuery;
import cn.wise.sc.cement.business.model.query.*;
import cn.wise.sc.cement.business.model.vo.*;
import cn.wise.sc.cement.business.service.IEquipmentService;
import cn.wise.sc.cement.business.service.ISysUserService;
import cn.wise.sc.cement.business.util.ExcelUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author ztw
 * @since 2020-09-01
 */
@Service
public class EquipmentServiceImpl extends ServiceImpl<EquipmentMapper, Equipment> implements IEquipmentService {

	@Resource
	private EquipmentMapper equipmentMapper;
	@Autowired
	private ISysUserService userService;
	@Resource
	private EquipmentTestMapper testMapper;
	@Autowired
	private CommonServiceImpl commonService;
	@Resource
	private EquipmentTroubleshootingMapper troubleshootingMapper;
	@Resource
	private EquipmentScrapMapper scrapMapper;
	@Resource
	private EquipmentUseMapper useMapper;

	/**
	 * 设备分页查询
	 * @param pageQuery
	 * @param brand
	 * @param supplierId
	 * @param name
	 * @return
	 */
	@Override
	public BaseResponse<IPage<EquipmentVo>> getPage(PageQuery pageQuery, String brand, Integer supplierId, String name) {
		Map<String, Object> params = new HashMap<>();
		params.put("brand", brand);
		params.put("supplierId", supplierId);
		params.put("name", name);
		Page<EquipmentVo> page = new Page<>(pageQuery.getPageNo(), pageQuery.getPageSize());
		IPage<EquipmentVo> pages = equipmentMapper.getPage(page,params);
		return BaseResponse.okData(pages);
	}

	/**
	 * 设备列表导出
	 * @param brand
	 * @param supplierId
	 * @param name
	 * @param fileName
	 * @param response
	 */
	@Override
	public void export(String brand, Integer supplierId, String name, String fileName, HttpServletResponse response) {
		Map<String, Object> params = new HashMap<>();
		params.put("brand", brand);
		params.put("supplierId", supplierId);
		List<Map<String, Object>> list= equipmentMapper.exportList(params);
		if (!CollectionUtils.isEmpty(list)) {
			Map<String, Object> map = list.get(0);
			String[] headers = new String[map.size()];
			headers[0] = "序号";
			headers[1] = "设备名称";
			headers[2] = "设备编号";
			headers[3] = "设备品牌";
			headers[4] = "规格/型号";
			headers[5] = "供应商";
			headers[6] = "位置";
			headers[7] = "购买日期";
			List<Object[]> datas = new ArrayList<>(list.size());
			for (Map<String, Object> m : list) {
				Object[] objects = new Object[headers.length];
				for (int j = 0; j < headers.length; j++) {
					String obj = m.get(headers[j]).toString();
					if(j==0){
						obj = obj.split("\\.")[0];
					}
					objects[j] = obj;
				}
				datas.add(objects);
			}
			ExcelUtil.excelExport(
					fileName == null || fileName.trim().length() <= 0 ? "设备列表": fileName, headers,
					datas, response);
		}
	}

	/**
	 * 新增设备
	 * @param query
	 * @return
	 */
	@Override
	@Transactional
	public BaseResponse<Equipment> create(EquipmentQuery query) {
		if (StringUtils.isEmpty(query.getName())) {
			return BaseResponse.errorMsg("设备名称不能为空!");
		}
		QueryWrapper<Equipment> qw = new QueryWrapper<>();
		qw.eq("name", query.getName());
		int count = equipmentMapper.selectCount(qw);
		if (count > 0) {
			return BaseResponse.errorMsg(query.getName() + "已存在");
		}
		Equipment equipment = new Equipment();
		BeanUtils.copyProperties(query,equipment);
		equipment.setStatus(1).setCreateTime(LocalDateTime.now());
		equipmentMapper.insert(equipment);
		return BaseResponse.okData(equipment);
	}

	/**
	 * 修改设备
	 * @param query
	 * @return
	 */
	@Transactional
	@Override
	public BaseResponse<Equipment> update(EquipmentQuery query) {
		if (StringUtils.isEmpty(query.getName())) {
			return BaseResponse.errorMsg("设备名称不能为空!");
		}
		QueryWrapper<Equipment> qw = new QueryWrapper<>();
		qw.eq("name", query.getName());
		qw.ne("id", query.getId());
		int count = equipmentMapper.selectCount(qw);
		if (count > 0) {
			return BaseResponse.errorMsg(query.getName() + "已存在");
		}
		Equipment equipment = new Equipment();
		BeanUtils.copyProperties(query,equipment);
		equipmentMapper.updateById(equipment);
		return BaseResponse.okData(equipment);
	}

	/**
	 * 获取设备详情
	 * @param id
	 * @return
	 */
	@Override
	public  BaseResponse<EquipmentVo> getDetail(Integer id){
		EquipmentVo equipmentVo = equipmentMapper.getDetail(id);
		if(equipmentVo == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		return  BaseResponse.okData(equipmentVo);

	}



	/**
	 * 设备检定 保存设备检定表信息和附件表信息，附件表的entity_type为3
	 * @param query （不传id为添加记录，传id为修改记录）
	 * @return
	 */
	@Transactional
	@Override
	public  BaseResponse<String> test(EquipmentTestQuery query){
		if(query== null || query.getEquipmentId()==null){
			return BaseResponse.errorMsg("参数错误!");
		}
//		EquipmentVo equipmentVo = equipmentMapper.getDetail(query.getEquipmentId());
		Equipment equipment = equipmentMapper.selectById(query.getEquipmentId());
		if(equipment == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		EnclosureQuery enclosureQuery = query.getEnclosureQuery();
		EquipmentTest equipmentTest = new EquipmentTest();
		BeanUtils.copyProperties(query, equipmentTest);
		equipmentTest.setUserId(userService.getLoginUser().getId())
				.setCreateTime(LocalDateTime.now());
		testMapper.insert(equipmentTest);
		equipment.setTestDate(equipmentTest.getTestDate());
		equipment.setCheckDate(equipmentTest.getCheckDate());
		equipment.setPeriodDate(equipmentTest.getPeriodDate());
		equipment.setTryResult(equipmentTest.getTryResult());
		equipment.setEnclosureUrl(enclosureQuery.getEnclosureUrl());
		equipment.setAlias(enclosureQuery.getAlias());
		equipment.setExtName(enclosureQuery.getExtName());
		equipmentMapper.updateById(equipment);
		//保存校核附件信息
		List<EnclosureQuery> enclosureQueryList = new ArrayList<>();
		enclosureQueryList.add(enclosureQuery);
		Boolean ref = commonService.saveEntityEnclosure(EntityEnclosure.EntityType.EQUIPMENT_TEST, equipmentTest.getId(), enclosureQueryList);
		if(!ref){
			return BaseResponse.errorMsg("保存保存校核附件失败!");
		}
		return  BaseResponse.okData("检定完成");
	}

	/**
	 *  设备检定分页查询
	 * @param pageQuery
	 * @param equipmentId
	 * @return
	 */
	@Override
	public BaseResponse<IPage<EquipmentTestVo>> getTestPage(PageQuery pageQuery, Integer equipmentId) {
		Map<String, Object> params = new HashMap<>();
		params.put("equipmentId", equipmentId);
		Page<EquipmentTestVo> page = new Page<>(pageQuery.getPageNo(), pageQuery.getPageSize());
		IPage<EquipmentTestVo> pages = testMapper.getPage(page,params);
		List<EquipmentTestVo> list = page.getRecords();
		if(list != null && list.size()>0){
			for(EquipmentTestVo equipmentTestVo : list){
				List<EntityEnclosure> entityEnclosureList = commonService.getEnclosureList
						(EntityEnclosure.EntityType.EQUIPMENT_TEST, equipmentTestVo.getId());
				String enclosure = "";
				if(entityEnclosureList != null && entityEnclosureList.size()>0){
					for(EntityEnclosure entityEnclosure : entityEnclosureList){
						String fileName = entityEnclosure.getAlias()+entityEnclosure.getExtName();
						enclosure = enclosure.equals("")?fileName:enclosure+"、"+fileName;
					}
				}
				equipmentTestVo.setEnclosure(enclosure);
			}
		}
		return BaseResponse.okData(pages);
	}

	/**
	 * 设备检定详情
	 * @param id 设备检定表id
	 * @return
	 */
	@Override
	public BaseResponse<EquipmentTestVo> getTestDetail(Integer id){
		EquipmentTestVo equipmentTestVo = testMapper.getTestDetail(id);
		if(equipmentTestVo == null || equipmentTestVo.getId() == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		List<EntityEnclosure> entityEnclosureList = commonService.getEnclosureList
				(EntityEnclosure.EntityType.EQUIPMENT_TEST, equipmentTestVo.getId());

		List<EnclosureQuery> enclosureQueryList = new ArrayList<>();
		for(EntityEnclosure entityEnclosure : entityEnclosureList){
			EnclosureQuery enclosureQuery = new EnclosureQuery();
			enclosureQuery.setAlias(entityEnclosure.getAlias())
					.setEnclosureUrl(entityEnclosure.getEnclosureUrl())
					.setExtName(entityEnclosure.getExtName());
			enclosureQueryList.add(enclosureQuery);
		}
		equipmentTestVo.setEnclosureQueryList(enclosureQueryList);
		return BaseResponse.okData(equipmentTestVo);
	}




	/**
	 * 设备故障维修登记
	 * @param query （不传id为添加记录，传id为修改记录）
	 * @return
	 */
	@Transactional
	@Override
	public  BaseResponse<String> troubleshooting(EquipmentTroubleshootingQuery query){
		if(query== null || query.getEquipmentId()==null){
			return BaseResponse.errorMsg("参数错误!");
		}
		EquipmentVo equipmentVo = equipmentMapper.getDetail(query.getEquipmentId());
		if(equipmentVo == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		EquipmentTroubleshooting equipmentTroubleshooting = new EquipmentTroubleshooting();
		BeanUtils.copyProperties(query, equipmentTroubleshooting);
		if(query.getId() == null){
			equipmentTroubleshooting.setUserId(userService.getLoginUser().getId())
					.setCreateTime(LocalDateTime.now());
			troubleshootingMapper.insert(equipmentTroubleshooting);
		}else{
			troubleshootingMapper.updateById(equipmentTroubleshooting);
		}
		return  BaseResponse.okData("故障维修登记完成");
	}


	/**
	 *  设备故障维修登记分页查询
	 * @param pageQuery
	 * @param equipmentId
	 * @return
	 */
	@Override
	public BaseResponse<IPage<EquipmentTroubleshootingVo>> getTroubleshootingPage(PageQuery pageQuery, Integer equipmentId) {
		Map<String, Object> params = new HashMap<>();
		params.put("equipmentId", equipmentId);
		Page<EquipmentTroubleshootingVo> page = new Page<>(pageQuery.getPageNo(), pageQuery.getPageSize());
		IPage<EquipmentTroubleshootingVo> pages = troubleshootingMapper.getPage(page,params);
		return BaseResponse.okData(pages);
	}

	/**
	 *  维修登记详情
	 * @param id 设备故障维修登记表id
	 * @return
	 */
	@Override
	public BaseResponse<EquipmentTroubleshootingVo> getTroubleshootingDetail(Integer id){
		EquipmentTroubleshootingVo equipmentTroubleshootingVo = troubleshootingMapper.getTroubleshootingDetail(id);
		if(equipmentTroubleshootingVo == null || equipmentTroubleshootingVo.getId() == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		return BaseResponse.okData(equipmentTroubleshootingVo);
	}

	/**
	 * 设备报废申请
	 * @param query
	 * @return
	 */
	@Override
	public  BaseResponse<String> scrap(EquipmentScrapQuery query){
		if(query== null || query.getEquipmentId()==null){
			return BaseResponse.errorMsg("参数错误!");
		}
		EquipmentVo equipmentVo = equipmentMapper.getDetail(query.getEquipmentId());
		if(equipmentVo == null){
			return BaseResponse.errorMsg("信息错误!");
		}

		EquipmentScrap equipmentScrap = new EquipmentScrap();
		BeanUtils.copyProperties(query, equipmentScrap);
		if(query.getId() == null){
			equipmentScrap.setUserId(userService.getLoginUser().getId())
					.setStatus(0)
					.setCreateTime(LocalDateTime.now());
			scrapMapper.insert(equipmentScrap);
		}else{
			scrapMapper.updateById(equipmentScrap);
		}
		return  BaseResponse.okData("设备报废申请完成");
	}

	/**
	 * 报废审批
	 * @param query
	 * @return
	 */
	@Override
	@Transactional
	public  BaseResponse<String> scrapApproval(EquipmentScrapApprovalQuery query){
		if(query.getId()== null){
			return BaseResponse.errorMsg("参数错误!");
		}
		EquipmentScrap equipmentScrap = scrapMapper.selectById(query.getId());
		if(equipmentScrap == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		equipmentScrap.setStatus(query.getIsAgree());
		scrapMapper.updateById(equipmentScrap);
		if(query.getIsAgree() == 1){
			Equipment equipment = equipmentMapper.selectById(equipmentScrap.getEquipmentId());
			if(equipment == null){
				return BaseResponse.errorMsg("信息错误!");
			}
			equipment.setStatus(3);//报废状态
			equipmentMapper.updateById(equipment);
		}
		return  BaseResponse.okData("审批完成");
	}

	/**
	 * 报废申请列表
	 * @param pageQuery
	 * @param brand
	 * @param supplierId
	 * @param name
	 * @return
	 */
	@Override
	public BaseResponse<IPage<EquipmentScrapVo>> getScrapApprovalPage(PageQuery pageQuery,
	                                                        String brand, Integer supplierId, String name) {
		Map<String, Object> params = new HashMap<>();
		params.put("brand", brand);
		params.put("supplierId", supplierId);
		params.put("name", name);
		Page<EquipmentScrapVo> page = new Page<>(pageQuery.getPageNo(), pageQuery.getPageSize());
		IPage<EquipmentScrapVo> pages = scrapMapper.getPage(page,params);
		return BaseResponse.okData(pages);
	}


	/**
	 * 报废申请详情
	 * @param id 设备报废表id
	 * @return
	 */
	@Override
	@Transactional
	public BaseResponse<EquipmentScrapVo> getscrapApprovalDetail(Integer id){
		EquipmentScrapVo equipmentScrapVo = scrapMapper.getScrapDetail(id);
		if(equipmentScrapVo == null || equipmentScrapVo.getId() == null){
			return BaseResponse.errorMsg("信息错误!");
		}
		return BaseResponse.okData(equipmentScrapVo);
	}

	/**
	 * 设备使用记录分页
	 * @param pageQuery
	 * @param brand
	 * @param supplierId
	 * @param name
	 * @return
	 */
	@Override
	public BaseResponse<IPage<EquipmentUseVo>> getUsePage(PageQuery pageQuery,
	                                                                  String brand, Integer supplierId, String name) {
		Map<String, Object> params = new HashMap<>();
		params.put("brand", brand);
		params.put("supplierId", supplierId);
		params.put("name", name);
		Page<EquipmentUseVo> page = new Page<>(pageQuery.getPageNo(), pageQuery.getPageSize());
		IPage<EquipmentUseVo> pages = useMapper.getPage(page,params);
		return BaseResponse.okData(pages);
	}














































}
