package com.testor.common.util;

import java.util.HashMap;
import java.util.Map;

public class DangerousOperationValidator {

    public enum OperationType {
        HOT_WORK("动火作业"),
        HEIGHT_WORK("高处作业"),
        CONFINED_SPACE("有限空间"),
        TEMP_ELECTRICITY("临时用电"),
        FUMIGATION("熏蒸作业"),
        IN_OUT_WAREHOUSE("进出仓");

        private final String name;
        OperationType(String name) {
            this.name = name;
        }
        public String getName() {
            return name;
        }

        public static OperationType fromDictValue(String dictValue) {
            for (OperationType type : values()) {
                if (type.getName().equals(dictValue)) {
                    return type;
                }
            }
            return null;
        }
    }

    public enum DangerLevel {
        HIGH("高度危险"),
        MEDIUM("较大危险"),
        LOW("一般危险");

        private final String level;
        DangerLevel(String level) {
            this.level = level;
        }
        public String getLevel() {
            return level;
        }

        public static DangerLevel fromDictValue(String dictValue) {
            for (DangerLevel level : values()) {
                if (level.getLevel().equals(dictValue)) {
                    return level;
                }
            }
            return null;
        }
    }

    private static final Map<OperationType, Map<DangerLevel, Integer>> TIME_LIMITS = new HashMap<>();
    static {
        // 动火作业
        Map<DangerLevel, Integer> hotWorkMap = new HashMap<>();
        hotWorkMap.put(DangerLevel.HIGH, 8);
        hotWorkMap.put(DangerLevel.MEDIUM, 12);
        hotWorkMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.HOT_WORK, hotWorkMap);

        // 高处作业
        Map<DangerLevel, Integer> heightWorkMap = new HashMap<>();
        heightWorkMap.put(DangerLevel.HIGH, 12);
        heightWorkMap.put(DangerLevel.MEDIUM, 24);
        heightWorkMap.put(DangerLevel.LOW, 7 * 24);
        TIME_LIMITS.put(OperationType.HEIGHT_WORK, heightWorkMap);

        // 有限空间
        Map<DangerLevel, Integer> confinedSpaceMap = new HashMap<>();
        confinedSpaceMap.put(DangerLevel.HIGH, 8);
        confinedSpaceMap.put(DangerLevel.MEDIUM, 8);
        confinedSpaceMap.put(DangerLevel.LOW, 8);
        TIME_LIMITS.put(OperationType.CONFINED_SPACE, confinedSpaceMap);

        // 临时用电
        Map<DangerLevel, Integer> tempElectricityMap = new HashMap<>();
        tempElectricityMap.put(DangerLevel.HIGH, 8);
        tempElectricityMap.put(DangerLevel.MEDIUM, 8);
        tempElectricityMap.put(DangerLevel.LOW, 8);
        TIME_LIMITS.put(OperationType.TEMP_ELECTRICITY, tempElectricityMap);

        // 熏蒸作业
        Map<DangerLevel, Integer> fumigationMap = new HashMap<>();
        fumigationMap.put(DangerLevel.HIGH, 7 * 24);
        // 其他等级不设置限制
        TIME_LIMITS.put(OperationType.FUMIGATION, fumigationMap);

        // 进出仓
        Map<DangerLevel, Integer> inOutWarehouseMap = new HashMap<>();
        inOutWarehouseMap.put(DangerLevel.HIGH, 12);
        inOutWarehouseMap.put(DangerLevel.MEDIUM, 24);
        inOutWarehouseMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.IN_OUT_WAREHOUSE, inOutWarehouseMap);
    }

    public static String validateOperation(String workTypeDictValue, String workLevelDictValue, double actualHours) {
        // 转换为枚举类型
        OperationType type = OperationType.fromDictValue(workTypeDictValue);
        DangerLevel level = DangerLevel.fromDictValue(workLevelDictValue);

        // 如果类型或等级不在规则范围内，直接返回安全
        if (type == null || level == null) {
            return "在安全时间内";
        }

        // 获取该类型的时间限制映射
        Map<DangerLevel, Integer> levelMap = TIME_LIMITS.get(type);
        if (levelMap == null) {
            return "在安全时间内";
        }

        // 获取具体等级的时间限制
        Integer timeLimit = levelMap.get(level);

        // 如果没有找到该等级的限制，返回安全
        if (timeLimit == null) {
            return "在安全时间内";
        }

        // 检查实际时长是否超过限制
        int actualHoursInt = (int) Math.ceil(actualHours);
        if (actualHoursInt > timeLimit) {
            return String.format("%s-%s作业超过规定时间%d小时，规定最大时长为%d小时。",
                    type.getName(), level.getLevel(),
                    actualHoursInt - timeLimit, timeLimit);
        }

        return "在安全时间内";
    }
}