package com.testor.ddd.safetyControl.domain.equipment.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.testor.ddd.safetyControl.domain.equipment.entity.TSafeEquipmentRepo;
import com.testor.ddd.safetyControl.domain.equipment.service.EquipmentService;
import com.testor.ddd.safetyControl.interfaces.model.dto.equipment.TSafeEquipmentDTO;
import com.testor.ddd.safetyControl.interfaces.model.dto.equipment.TSafeEquipmentDTOParam;
import com.testor.ddd.safetyControl.interfaces.model.vo.equipment.EquipmentStatisticsVO;
import com.testor.ddd.safetyControl.interfaces.model.vo.equipment.TSafeEquipmentVO;
import com.tongtech.tfw.backend.common.biz.models.BaseResponse;
import com.tongtech.tfw.backend.common.biz.models.BaseResponseList;
import com.tongtech.tfw.backend.common.constants.enumeration.BaseStatusEnum;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

/**
 * 设备Service业务层处理
 * 
 * @author testor-framework
 * @date 2022-06-22 18:03:10
 */
@Service
public class EquipmentServiceImpl implements EquipmentService
{
    @Resource
    private TSafeEquipmentRepo tSafeEquipmentRepo;

    @Resource
    private TSafeEquipmentRepo equipmentRepo;

    /**
     * 检测 名称、编码 是否都唯一
     * @param id
     * @param name
     * @param code
     * @return true - 唯一； false - 不唯一
     */
    @Override
    public boolean checkUniqueness(String id, String equipmentTypeId, String name, String code){
        List<TSafeEquipmentDTO> equipmentDTOList = equipmentRepo.checkUniqueness(id, equipmentTypeId, name, code);
        if(equipmentDTOList == null || equipmentDTOList.size() == 0) {
            return true;
        }
        return false;
    }

    /**
     * 编辑空间
     *
     * @param equipmentDto
     * @return
     */
    @Transactional
    public BaseResponse editEquipment(String userId, String orgId, TSafeEquipmentDTO equipmentDto) {
        BaseResponse baseResponse = new BaseResponse<>();

        boolean checkResult = checkUniqueness(equipmentDto.getId(), equipmentDto.getEquipmentTypeId(), equipmentDto.getName(), equipmentDto.getCode());
        if (!checkResult) {
            baseResponse.setCode(BaseStatusEnum.UNKNOWN.code());
            baseResponse.setMsg("已存在相同名称/编码的数据");
            return baseResponse;
        }

        String dataId = equipmentDto.getId();
        if (StrUtil.isBlank(dataId)) { // 新增
            equipmentDto.setCreateBy(userId);
            equipmentDto.setOrgId(orgId);
            equipmentRepo.insert(equipmentDto);
            return baseResponse;
        }

        if (!orgId.equals(equipmentDto.getOrgId())) {
            baseResponse.setCode(BaseStatusEnum.UN_DATA.code());
            baseResponse.setMsg("当前用户无法操作该条数据");
            return baseResponse;
        }

        equipmentDto.setUpdateBy(userId);
        equipmentRepo.updateById(equipmentDto);
        return baseResponse;
    }

    @Override
    public BaseResponseList<TSafeEquipmentVO> listByPage(TSafeEquipmentDTOParam param){
        BaseResponseList<TSafeEquipmentVO>  baseResponseList = equipmentRepo.listPage(param);
        return baseResponseList;
    }

    @Override
    public TSafeEquipmentDTO findById(String id){
        return equipmentRepo.findById(id);
    }

    /**
     * 根据条件统计设备信息
     * @param orgId
     * @return
     */
    @Override
    public EquipmentStatisticsVO statisticsEquipment(String orgId){
        if(StrUtil.isBlank(orgId)) {
            return null;
        }
        TSafeEquipmentDTOParam countParam;
        Date nowDate = new Date();
        // 总数
        countParam = new TSafeEquipmentDTOParam();
        countParam.setOrgId(orgId);
        countParam.setIsSpecial("1");
        Integer total = tSafeEquipmentRepo.countEquipment(countParam);

        // 本年度待检数量
        countParam = new TSafeEquipmentDTOParam();
        countParam.setOrgId(orgId);
        countParam.setIsSpecial("1");
        DateTime yearBegin = DateUtil.beginOfYear(nowDate);
        DateTime yearEnd = DateUtil.endOfYear(nowDate);
        countParam.setValidityStartDate(com.testor.common.util.DateUtil.localDateTimeToDate(DateUtil.toLocalDateTime(yearBegin)));
        countParam.setValidityEndDate(com.testor.common.util.DateUtil.localDateTimeToDate(DateUtil.toLocalDateTime(yearEnd)));
        Integer checkNum = tSafeEquipmentRepo.countEquipment(countParam);

        // 即将过期数量
        countParam = new TSafeEquipmentDTOParam();
        countParam.setOrgId(orgId);
        countParam.setIsSpecial("1");
        DateTime endDate = DateUtil.offsetDay(nowDate, 30);
        countParam.setValidityStartDate(com.testor.common.util.DateUtil.localDateTimeToDate(DateUtil.toLocalDateTime(nowDate)));
        countParam.setValidityEndDate(com.testor.common.util.DateUtil.localDateTimeToDate(DateUtil.toLocalDateTime(endDate)));
        Integer expiringSoonNum = tSafeEquipmentRepo.countEquipment(countParam);

        // 已过期数量
        countParam = new TSafeEquipmentDTOParam();
        countParam.setOrgId(orgId);
        countParam.setIsSpecial("1");
        countParam.setValidityEndDate(com.testor.common.util.DateUtil.localDateTimeToDate(DateUtil.toLocalDateTime(nowDate)));
        Integer expiredNum = tSafeEquipmentRepo.countEquipment(countParam);

        EquipmentStatisticsVO equipmentStatisticsVO = new EquipmentStatisticsVO();
        equipmentStatisticsVO.setTotal(total);
        equipmentStatisticsVO.setCheckNum(checkNum);
        equipmentStatisticsVO.setExpiredNum(expiredNum);
        equipmentStatisticsVO.setExpiringSoonNum(expiringSoonNum);
        return equipmentStatisticsVO;
    }
}
