package com.testor.module.iam.service.impl;

import com.testor.biz.sys.user.model.domain.SysUser;
import com.testor.module.contractor.manage.dao.TContractorAllowDao;
import com.testor.module.contractor.manage.model.domain.TContractorAllow;
import com.testor.module.contractor.manage.model.dto.TContractorAllowDTO;
import com.testor.module.iam.dao.IamSysUserDao;
import com.testor.module.iam.dao.SysUserDao;
import com.testor.module.iam.model.domain.IamSysUser;
import com.testor.module.iam.service.IamSysUserService;
import com.testor.module.iam.service.SysUserService;
import com.tongtech.tfw.backend.common.models.supers.SuperServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;

@Service("SysUserService")
public class SysUserServiceImpl extends SuperServiceImpl<SysUserDao, SysUser> implements SysUserService {

    @Autowired
    private TContractorAllowDao tContractorAllowDao;

    /**
     * 获取有权限审批当前组织承包商数据的用户
     */
    public List<SysUser> getApprovalUsers(String orgId) {
        return baseMapper.selectApprovalUsersByOrgId(orgId);
    }

    /**
     * 获取有权限审批特定承包商记录的用户
     */
    @Transactional(readOnly = true)
    public List<SysUser> getApprovalUsersForContractor(String contractorId, String orgId) {
        // 首先验证承包商记录是否存在且可访问
        TContractorAllow contractor = tContractorAllowDao.selectById(contractorId);
        if (contractor == null) {
            throw new RuntimeException("承包商记录不存在");
        }

        // 检查当前用户是否有权限访问该承包商记录
        boolean hasAccess = baseMapper.checkOrgAccess(contractor.getOrgId(), orgId);
        if (!hasAccess) {
            throw new RuntimeException("无权限访问该承包商记录");
        }

        return baseMapper.selectApprovalUsersForContractor(contractorId, orgId);
    }

    /**
     * 验证用户是否有审批权限
     */
    public boolean validateApprovalPermission(String userId, String recordId, String orgId) {
        // 1. 检查用户是否有审批角色
        boolean hasRole = baseMapper.checkUserHasApprovalRole(userId);
        if (!hasRole) {
            return false;
        }

        // 2. 获取用户信息
        SysUser user = baseMapper.selectById(userId);
        if (user == null) {
            return false;
        }

        // 3. 检查用户组织权限
        boolean hasOrgAccess = baseMapper.checkOrgAccess(user.getOrgId(), orgId);
        if (!hasOrgAccess) {
            return false;
        }

        // 4. 检查记录访问权限
        TContractorAllow record = tContractorAllowDao.selectById(recordId);
        if (record == null) {
            return false;
        }

        return baseMapper.checkOrgAccess(record.getOrgId(), user.getOrgId());
    }

    /**
     * 验证用户是否有审批当前组织数据的权限
     */
    public boolean validateApprovalPermissionForOrg(String userId, String orgId) {
        // 1. 检查用户是否有审批角色
        boolean hasRole = baseMapper.checkUserHasApprovalRole(userId);
        if (!hasRole) {
            return false;
        }

        // 2. 获取用户信息
        SysUser user = baseMapper.selectById(userId);
        if (user == null) {
            return false;
        }

        // 3. 检查用户组织权限
        return baseMapper.checkOrgAccess(user.getOrgId(), orgId);
    }
}
