package com.testor.module.accident.model.template;

import com.testor.common.util.BeanConverUtil;
import com.testor.module.accident.model.domain.AccidentPersonnel;
import com.testor.module.accident.model.dto.AccidentReportDetail;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

public class BeanConverExtensionUtil {

    /**
     * 转换事故详情列表，并处理人员信息字段
     */
    public static List<AccidentReportDetailTemplate> converAccidentReportDetailList(
            List<AccidentReportDetail> sourceList) {
        if (CollectionUtils.isEmpty(sourceList)) {
            return Collections.emptyList();
        }

        return sourceList.stream()
                .map(BeanConverExtensionUtil::converAccidentReportDetail)
                .collect(Collectors.toList());
    }

    /**
     * 转换单个事故详情
     */
    public static AccidentReportDetailTemplate converAccidentReportDetail(AccidentReportDetail source) {
        if (source == null) {
            return null;
        }

        // 使用BeanConverUtil转换基本字段
        AccidentReportDetailTemplate template = BeanConverUtil.conver(source, AccidentReportDetailTemplate.class);

        // 手动处理日期字段
        if (source.getAccidentTime() != null) {
            java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            template.setAccidentTime(sdf.format(source.getAccidentTime()));
        }

        // 手动处理BigDecimal字段
        if (source.getDirectEconomicLoss() != null) {
            template.setDirectEconomicLoss(source.getDirectEconomicLoss().toString());
        }

        // 处理人员信息字段
        if (!CollectionUtils.isEmpty(source.getAccidentPersonnelList())) {
            template.setPersonnelBelongType(formatPersonnelBelongType(source.getAccidentPersonnelList()));
            template.setPersonnelAge(formatPersonnelAge(source.getAccidentPersonnelList()));
            template.setPersonnelSex(formatPersonnelSex(source.getAccidentPersonnelList()));
        } else {
            template.setPersonnelBelongType("无");
            template.setPersonnelAge("无");
            template.setPersonnelSex("无");
        }

        return template;
    }

    private static String formatPersonnelBelongType(List<AccidentPersonnel> personnelList) {
        if (com.baomidou.mybatisplus.core.toolkit.CollectionUtils.isEmpty(personnelList)) {
            return "";
        }

        List<String> belongTypes = new ArrayList<>();
        for (AccidentPersonnel person : personnelList) {
            String belongType = person.getBelongType();
            if (com.testor.common.core.utils.StringUtils.hasText(belongType)) {
                if ("0".equals(belongType)) {
                    belongTypes.add("承包商人员");
                } else if ("1".equals(belongType)) {
                    belongTypes.add("企业人员");
                } else {
                    belongTypes.add(belongType); // 如果是其他值，保持原样
                }
            }
        }

        return String.join(", ", belongTypes);
    }

    private static String formatPersonnelAge(List<AccidentPersonnel> personnelList) {
        return personnelList.stream()
                .map(person -> person.getAge() != null ? String.valueOf(person.getAge()) : "")
                .filter(StringUtils::hasText)
                .collect(Collectors.joining(", "));
    }

    private static String formatPersonnelSex(List<AccidentPersonnel> personnelList) {
        return personnelList.stream()
                .map(person -> {
                    String sex = person.getSex();
                    if ("0".equals(sex)) {
                        return "男";
                    } else if ("1".equals(sex)) {
                        return "女";
                    } else {
                        return sex != null ? sex : "";
                    }
                })
                .filter(StringUtils::hasText)
                .collect(Collectors.joining(", "));
    }
}