package com.testor.common.util;

import com.testor.module.hazard.model.domain.THazardWorkPlan;

import java.util.HashMap;
import java.util.Map;

public class DangerousOperationValidator {

    public enum OperationType {
        HOT_WORK("动火作业"),
        HEIGHT_WORK("高处作业"),
        CONFINED_SPACE("有限空间作业"),
        TEMP_ELECTRICITY("临时用电作业"),
        FUMIGATION("熏蒸作业"),
        IN_WAREHOUSE("进仓作业"),
        OUT_WAREHOUSE("出仓作业"),
        FLAG_WAREHOUSE("平仓作业"),
        IN_OUT_WAREHOUSE("进出仓作业"),
        CIRCULATION_FUMIGATION("环流熏蒸作业");

        private final String name;
        OperationType(String name) {
            this.name = name;
        }
        public String getName() {
            return name;
        }

        public static OperationType fromDictValue(String dictValue) {
            for (OperationType type : values()) {
                if (type.getName().equals(dictValue)) {
                    return type;
                }
            }
            return null;
        }
    }

    public enum DangerLevel {
        HIGH("高度危险"),
        MEDIUM("较大危险"),
        LOW("一般危险");

        private final String level;
        DangerLevel(String level) {
            this.level = level;
        }
        public String getLevel() {
            return level;
        }

        public static DangerLevel fromDictValue(String dictValue) {
            for (DangerLevel level : values()) {
                if (level.getLevel().equals(dictValue)) {
                    return level;
                }
            }
            return null;
        }
    }

    private static final Map<OperationType, Map<DangerLevel, Integer>> TIME_LIMITS = new HashMap<>();
    static {
        // 动火作业
        Map<DangerLevel, Integer> hotWorkMap = new HashMap<>();
        hotWorkMap.put(DangerLevel.HIGH, 8);
        hotWorkMap.put(DangerLevel.MEDIUM, 12);
        hotWorkMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.HOT_WORK, hotWorkMap);

        // 高处作业
        Map<DangerLevel, Integer> heightWorkMap = new HashMap<>();
        heightWorkMap.put(DangerLevel.HIGH, 12);
        heightWorkMap.put(DangerLevel.MEDIUM, 24);
        heightWorkMap.put(DangerLevel.LOW, 7 * 24);
        TIME_LIMITS.put(OperationType.HEIGHT_WORK, heightWorkMap);

        // 有限空间
        Map<DangerLevel, Integer> confinedSpaceMap = new HashMap<>();
        confinedSpaceMap.put(DangerLevel.HIGH, 8);
        confinedSpaceMap.put(DangerLevel.MEDIUM, 8);
        confinedSpaceMap.put(DangerLevel.LOW, 8);
        TIME_LIMITS.put(OperationType.CONFINED_SPACE, confinedSpaceMap);

        // 临时用电
        Map<DangerLevel, Integer> tempElectricityMap = new HashMap<>();
        tempElectricityMap.put(DangerLevel.HIGH, 8);
        tempElectricityMap.put(DangerLevel.MEDIUM, 8);
        tempElectricityMap.put(DangerLevel.LOW, 8);
        TIME_LIMITS.put(OperationType.TEMP_ELECTRICITY, tempElectricityMap);

        // 熏蒸作业
        Map<DangerLevel, Integer> fumigationMap = new HashMap<>();
        fumigationMap.put(DangerLevel.HIGH, 7 * 24);
        // 其他等级不设置限制
        TIME_LIMITS.put(OperationType.FUMIGATION, fumigationMap);

        // 环流熏蒸
        Map<DangerLevel, Integer> circulationFumigationMap = new HashMap<>();
        circulationFumigationMap.put(DangerLevel.HIGH, 7 * 24);
        // 其他等级不设置限制
        TIME_LIMITS.put(OperationType.CIRCULATION_FUMIGATION, circulationFumigationMap);

        // 进仓
        Map<DangerLevel, Integer> inWarehouseMap = new HashMap<>();
        inWarehouseMap.put(DangerLevel.HIGH, 12);
        inWarehouseMap.put(DangerLevel.MEDIUM, 24);
        inWarehouseMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.IN_WAREHOUSE, inWarehouseMap);

        // 出仓
        Map<DangerLevel, Integer> outWarehouseMap = new HashMap<>();
        outWarehouseMap.put(DangerLevel.HIGH, 12);
        outWarehouseMap.put(DangerLevel.MEDIUM, 24);
        outWarehouseMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.OUT_WAREHOUSE, outWarehouseMap);

        // 平仓
        Map<DangerLevel, Integer> flagWarehouseMap = new HashMap<>();
        flagWarehouseMap.put(DangerLevel.HIGH, 12);
        flagWarehouseMap.put(DangerLevel.MEDIUM, 24);
        flagWarehouseMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.FLAG_WAREHOUSE, flagWarehouseMap);

        // 进出仓
        Map<DangerLevel, Integer> inOutWarehouseMap = new HashMap<>();
        inOutWarehouseMap.put(DangerLevel.HIGH, 12);
        inOutWarehouseMap.put(DangerLevel.MEDIUM, 24);
        inOutWarehouseMap.put(DangerLevel.LOW, 72);
        TIME_LIMITS.put(OperationType.IN_OUT_WAREHOUSE, inOutWarehouseMap);
    }

    public static String validateOperation(String workTypeDictValue, String workLevelDictValue, double actualHours, THazardWorkPlan plan) {
        // 转换为枚举类型
        OperationType type = OperationType.fromDictValue(workTypeDictValue);
        DangerLevel level = DangerLevel.fromDictValue(workLevelDictValue);

        // 如果类型或等级不在规则范围内，直接返回安全
        if (type == null || level == null) {
            return "在安全时间内";
        }

        // 获取该类型的时间限制映射
        Map<DangerLevel, Integer> levelMap = TIME_LIMITS.get(type);
        if (levelMap == null) {
            return "在安全时间内";
        }

        // 特殊处理：进出仓、进仓、出仓作业的较大危险级别
        Integer timeLimit = getAdjustedTimeLimit(type, level, plan);

        // 如果没有找到该等级的限制，返回安全
        if (timeLimit == null) {
            return "在安全时间内";
        }

        // 检查实际时长是否超过限制
        int actualHoursInt = (int) Math.ceil(actualHours);
        if (actualHoursInt > timeLimit) {
            return String.format("%s-%s作业超过规定时长%d小时，规定最大时长为%d小时。",
                    type.getName(), level.getLevel(),
                    actualHoursInt - timeLimit, timeLimit);
        }

        return "在安全时间内";
    }

    /**
     * 获取调整后的时间限制（处理特殊情况）
     */
    private static Integer getAdjustedTimeLimit(OperationType type, DangerLevel level, THazardWorkPlan plan) {
        // 获取默认的时间限制
        Map<DangerLevel, Integer> levelMap = TIME_LIMITS.get(type);
        if (levelMap == null) {
            return null;
        }

        Integer timeLimit = levelMap.get(level);
        if (timeLimit == null) {
            return null;
        }

        //如果是非北良的熏蒸作业高度为1小时,如果是北良的为7天
        if (level == DangerLevel.HIGH){
            if(type == OperationType.FUMIGATION){
                if(plan.isBl()){
                    timeLimit = 7 * 24;
                }else{
                    timeLimit = 1;
                }
            }
        }

        // 特殊规则：进出仓、进仓、出仓作业的较大危险级别
        if (level == DangerLevel.MEDIUM &&
                (type == OperationType.IN_OUT_WAREHOUSE ||
                        type == OperationType.IN_WAREHOUSE ||
                        type == OperationType.OUT_WAREHOUSE ||
                        type == OperationType.FLAG_WAREHOUSE)) {

            // 检查是否满足特殊条件：isFullyInvested为1或者isFull为1
            boolean isSpecialCondition = isSpecialConditionMet(plan);

            if (isSpecialCondition) {
                // 满足条件：时间限制为24小时
                timeLimit = 24;
            } else {
                // 不满足条件：时间限制为72小时
                timeLimit = 72;
            }
        }

        return timeLimit;
    }

    /**
     * 检查是否满足特殊条件：isFullyInvested为1或者isFull为1
     */
    private static boolean isSpecialConditionMet(THazardWorkPlan plan) {
        if (plan == null) {
            return false;
        }
        String isEscalated = plan.getIsEscalated();
        String isFull = plan.getIsFull();
        return "1".equals(isEscalated) || "1".equals(isFull);
    }

}