package com.testor.ddd.safetyControl.domain.space.service.impl;

import cn.hutool.core.util.StrUtil;
import com.testor.ddd.safetyControl.domain.space.entity.TSafeSpaceRepo;
import com.testor.ddd.safetyControl.domain.space.service.SpaceService;
import com.testor.ddd.safetyControl.infrastructure.uitls.SpaceLevelEnum;
import com.testor.ddd.safetyControl.interfaces.model.dto.space.TSafeSpaceDTO;
import com.testor.ddd.safetyControl.interfaces.model.dto.space.TSafeSpaceDTOParam;
import com.testor.ddd.safetyControl.interfaces.model.vo.space.TSafeSpaceVO;
import com.tongtech.tfw.backend.common.biz.models.BaseResponse;
import com.tongtech.tfw.backend.common.biz.models.BaseResponseList;
import com.tongtech.tfw.backend.common.constants.enumeration.BaseStatusEnum;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

@Service
public class SpaceServiceImpl implements SpaceService {

    @Resource
    private TSafeSpaceRepo spaceRepo;

    /**
     * 检测 名称、编码 是否都唯一
     * @param id
     * @param bizId level = 1 时 此值为space_type_id；2 :space_id
     * @param level 等级 1-车间场所名称；2-作业区域
     * @param name
     * @param code
     * @return true - 唯一； false - 不唯一
     */
    public boolean checkUniqueness(String id, String bizId, Integer level, String name, String code){
        List<TSafeSpaceDTO> spaceDTOList = spaceRepo.checkUniqueness(id, bizId, level, name, code);
        if(spaceDTOList == null || spaceDTOList.size() == 0) {
            return true;
        }
        return false;
    }

    /**
     * 编辑空间
     *
     * @param spaceDto
     * @return
     */
    @Transactional
    public BaseResponse editSpace(String userId, String orgId, TSafeSpaceDTO spaceDto) {
        BaseResponse baseResponse = new BaseResponse<>();

        Integer level = spaceDto.getLevel();
        String bizId = null;
        if(level == SpaceLevelEnum.workshop.getCode().intValue()) {
            bizId = spaceDto.getSpaceTypeId();
        }
        if(level == SpaceLevelEnum.workarea.getCode().intValue()) {
            bizId = spaceDto.getSpaceId();
        }
        if(StrUtil.isBlank(bizId)) {
            baseResponse.setCode(BaseStatusEnum.UN_DATA.code());
            baseResponse.setMsg("参数缺失");
            return baseResponse;
        }

        boolean checkResult = checkUniqueness(spaceDto.getId(), bizId, level, spaceDto.getName(), spaceDto.getCode());
        if (!checkResult) {
            baseResponse.setCode(BaseStatusEnum.UNKNOWN.code());
            baseResponse.setMsg("已存在相同名称/编码的数据");
            return baseResponse;
        }

        String dataId = spaceDto.getId();
        if (StrUtil.isBlank(dataId)) { // 新增
            spaceDto.setCreateBy(userId);
            spaceDto.setOrgId(orgId);
            spaceRepo.insert(spaceDto);
            return baseResponse;
        }

        if (!orgId.equals(spaceDto.getOrgId())) {
            baseResponse.setCode(BaseStatusEnum.UN_DATA.code());
            baseResponse.setMsg("当前用户无法操作该条数据");
            return baseResponse;
        }

        spaceDto.setUpdateBy(userId);
        spaceRepo.updateById(spaceDto);
        return baseResponse;
    }

    public BaseResponseList<TSafeSpaceVO> listByPage(TSafeSpaceDTOParam param){
        if(param.getLevel() == null) {
            return null;
        }
        Integer level = param.getLevel();
        BaseResponseList<TSafeSpaceVO> baseResponseList=new BaseResponseList<>();
        if(level == SpaceLevelEnum.workshop.getCode().intValue()) {
            baseResponseList = spaceRepo.listPageByWorkshop(param);
        }
        if(level == SpaceLevelEnum.workarea.getCode().intValue()) {
            baseResponseList = spaceRepo.listPageByWorkArea(param);
        } else {

        }
        return baseResponseList;
    }

    /**
     * 根据编码查询
     * @param code
     * @return
     */
    public List<TSafeSpaceVO> listByCode(String code){
        if(StrUtil.isBlank(code)) {
            return null;
        }
        List<TSafeSpaceVO> baseResponseList= spaceRepo.listByCode(code);

        return baseResponseList;
    }

    public TSafeSpaceDTO findById(String id){
        return spaceRepo.findById(id);
    }

    @Override
    public List<TSafeSpaceDTO> myList(String orgId) {
        TSafeSpaceDTOParam param=new TSafeSpaceDTOParam();
        param.setOrgId(orgId);
        return spaceRepo.listAllEntity(param);
    }

    @Override
    public List<TSafeSpaceVO> listSafeSpaceAll(TSafeSpaceDTOParam param) {
        return spaceRepo.listSafeSpaceAll(param);
    }
}
